<?php declare(strict_types = 1);

namespace Tags\AdminModule\Components;

use Blog\Model\UI;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Utils\ArrayHash;
use Tags\FrontModule\Model\TagsCache;
use Tags\Model\Entities\Tag;
use Tags\Model\Entities\TagText;

class TagForm extends BaseControl
{
	public ?Tag $tag = null;

	public function __construct(
		protected TagsCache $tagsCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('title', 'default.title')
			->setRequired()
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);

		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[]           = $this->formSuccess(...);
		$form->onSuccess['redirect'] = function(): never {
			$this->presenter->redirect('Default:');
		};

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$texts      = [];
			$langValues = $form->convertMultilangValuesToArray();

			if ($this->tag) {
				$tag          = $this->tag;
				$texts        = $tag->texts->toArray();
				$flashMessage = 'tags.tagForm.edited';
			} else {
				$tag          = new Tag('', $values->isPublished);
				$flashMessage = 'tags.tagForm.added';
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l])) {
					$texts[$l] = new TagText($tag, $l, $v['title']);
				}

				$texts[$l]->title = $v['title'];
				$this->em->persist($texts[$l]);
			}

			$tag->title       = '';
			$tag->isPublished = $values->isPublished;
			$tag->setTexts($texts);

			$this->em->persist($tag);
			$this->em->flush();
			$this->tagsCache->clearCache();

			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$this->presenter->flashMessageDanger('default.error');
			$form->addError($e->getMessage());
		}

		$this->presenter->redrawControl('flashes');
	}

	public function setTag(string|int $id): void
	{
		$this->tag = $this->em->getRepository(Tag::class)->find($id);

		if ($this->tag) {
			$d    = [];
			$form = $this['form'];
			$form->setDefaults([
				'isPublished' => $this->tag->isPublished,
			]);

			if (!$this->tag->texts->count()) {
				$d['title']['cs'] = $this->tag->title;
			} else {
				foreach ($this->tag->texts as $l => $text) {
					$d['title'][$l] = $text->title;
				}
			}
			$form->setDefaults($d);
		} else {
			$this->presenter->error();
		}
	}
}
