<?php declare(strict_types = 1);

namespace Teams\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Teams\Model\Members;

class MembersGrid extends BaseControl
{
	/** @var Members */
	protected $membersService;

	public function __construct(Members $members)
	{
		$this->membersService = $members;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->membersService->getEr()->createQueryBuilder('g')->addSelect('gr')->join('g.groups', 'gr')
			->orderBy('g.position', 'ASC');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('membersGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:editMember');
		$grid->addColumnText('group', 'default.group')->setRenderer(function($row) {
			return implode(', ', array_map(function($v) { return $v->title; }, $row->groups->toArray()));
		});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter

		// Actions
		$grid->addAction('edit', '', 'Default:editMember')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->membersService->remove($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->membersService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->membersService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
	}
}
