<?php declare(strict_types = 1);

namespace Teams\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Teams\Model\Entities\Member;
use Teams\Model\Groups;
use Teams\Model\Members;

class MemberForm extends TemplateReaderControl
{
	/** @var Member */
	public $entity;

	/** @var Members */
	protected $membersService;

	/** @var Groups */
	protected $groupsService;

	public function __construct(Members $members, Groups $groups)
	{
		$this->membersService = $members;
		$this->groupsService  = $groups;
	}

	public function render()
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->membersService->getTemplatesDir());
		$this->templateReader->setTranslateKey('teams.templateMember');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$groups = [];
		foreach ($this->groupsService->getEr()->findBy([], ['title' => 'ASC']) as $g) {
			$arr = [];
			$t   = $g;
			while ($t->parent) {
				$t     = $t->parent;
				$arr[] = $t->title;
			}
			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}
		asort($groups);

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addCheckboxList('group', 'default.group', $groups)->setRequired();
		$form->addLangsSelect('lang', 'default.language');
		$form->addSelect('template', $this->t('default.templates'), $templates)->setPrompt($this->t('default.choose'))
			->setTranslator(null)->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->entity) {
				$member       = $this->entity;
				$flashMessage = 'teams.memberForm.edited';
			} else {
				$member       = new Member($values->title, $values->template);
				$flashMessage = 'teams.memberForm.added';
			}

			$member->title       = $values->title;
			$member->isPublished = $values->isPublished;
			$member->lang        = $values->lang;
			$member->template    = $values->template;
			$member->setTexts($form->getComponent('component')->getValues(true));
			$member->groups = $values->group ? array_map(function($id) {
				return $this->groupsService->get((int) $id);
			}, $values->group) : [];

			$this->em->persist($member)->flush();
			$form->addCustomData('memberId', $member->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setEntity($id)
	{
		$this->entity = $this->membersService->get($id);
		if ($this->entity) {
			$m    = $this->entity;
			$form = $this['form'];
			$form->setDefaults([
				'title'       => $m->title,
				'isPublished' => $m->isPublished,
				'lang'        => $m->lang,
			]);
			$d = [];
			foreach ($m->groups as $g) {
				if (array_key_exists($g->getId(), $form['group']->getItems())) {
					$d[] = $g->getId();
				}
			}

			$form['group']->setDefaultValue($d);

			if ($m->template && array_key_exists($m->template, $form['template']->getItems())) {
				$form['template']->setDefaultValue($m->template);

				if ($this->httpRequest->getQuery('do') != 'memberForm-loadInputs') {
					$texts = $this->entity->getTexts();
					$texts = is_array($texts) ? $texts : $texts->toArray();
					$this->templateReader->loadTemplateComponents($this['form']['component'], $this->httpRequest->getPost('template') ?: $this->entity->template, $texts);
					$this->templateReader->setDefaults($this['form']['component'], $texts, $this->entity->template);
				}
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
