<?php declare(strict_types = 1);

namespace Teams\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Mapping\Annotation as Gedmo;
use Doctrine\ORM\Mapping as ORM;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="teams__group")
 * @ORM\Entity(repositoryClass="Gedmo\Tree\Entity\Repository\NestedTreeRepository")
 */
class Group
{
	use TId;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255)
	 */
	public $title;

	/**
	 * @var Member
	 * @ORM\ManyToMany(targetEntity="Member", mappedBy="groups")
	 */
	public $members;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var Group[]
	 * @ORM\OneToMany(targetEntity="Group", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	private $children;

	/**
	 * @var Group
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Group", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	private $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	private $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	private $rgt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Group")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private $root;

	public function __construct($title)
	{
		$this->title       = $title;
		$this->isPublished = 1;
		$this->members     = new ArrayCollection();
		$this->children    = new ArrayCollection();
	}

	/******
	 * === Members
	 */

	/**
	 * @return ArrayCollection|Member[]
	 */
	public function getMembers() { return $this->members; }

	public function addMember($member)
	{
		if ($this->members->contains($member)) {
			return true;
		}

		return $this->members->add($member);
	}

	public function removeMember($member)
	{
		if ($this->members->contains($member)) {
			return $this->members->removeElement($member);
		}

		return true;
	}

	/******
	 * === Group
	 */

	public function getRoot()
	{
		return $this->root;
	}

	public function setParent($parent = null)
	{
		$this->parent = $parent;
	}

	public function getParent()
	{
		return $this->parent;
	}

	public function getChildren() { return $this->children; }

	public function getLevel() { return $this->lvl; }
}
