<?php declare(strict_types = 1);

namespace Ulozenka\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\Event;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Ulozenka\Model\OrderApiService;
use Ulozenka\Model\OrdersExported;

class ExpeditionSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected OrdersExported  $ordersExported,
		protected Translator      $translator,
		protected OrderApiService $apiService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			Orders::class . '::processOrdersForExpedition' => 'processOrdersForExpedition',
			Orders::class . '::sendExpeditionData'         => 'sendExpeditionData',
			Orders::class . '::generateLabel'              => 'generateLabel',
		];
	}

	public function processOrdersForExpedition(Event $event): void
	{
		$orderExpeditions = [];
		$ids              = $this->filterOrders($event->data['data']);

		if ($ids !== []) {
			foreach ($this->ordersExported->getOrdersNotExported($ids) as $id => $expedition) {
				$orderExpeditions[$id] = new OrderExpedition($expedition->getOrder(), $this->translator->translate('ulozenka.name'), $expedition);
			}

			$event->data['expedition'] += $orderExpeditions;
		}
	}

	public function sendExpeditionData(Event $event): void
	{
		$ids      = $this->filterOrders($event->data['data']);
		$quantity = $event->data['quantity'] ?? 1;

		if ($ids !== []) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			if ($orders !== []) {
				$result = $this->apiService->generateOrderOverApi($orders, $quantity);

				$event->data['okCount']    += $result['ok'];
				$event->data['errorCount'] += $result['error'];
			}
		}
	}

	public function generateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids !== []) {
			$orders = $this->ordersExported->getOrdersExported($ids);

			if ($orders !== []) {
				$result = $this->apiService->generateLabelPdf($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$event->data['files'][] = $file;
					}
					$event->data['okCount'] += $result['ok'];
				} else {
					$event->data['errorCount'] += $result['error'];
				}
			}
		}
	}

	protected function filterOrders(array $data): array
	{
		$ids = [];
		foreach ($data as $v) {
			if ($v['service'] === 'ulozenka') {
				$ids[] = $v['id'];
			}
		}

		return $ids;
	}
}
