<?php declare(strict_types = 1);

namespace Ulozenka\Model\Entities;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Listeners\CarrierListener;
use Ulozenka\Model\UlozenkaConfig;

#[ORM\Table(name: 'ulozenka__order')]
#[ORM\Entity]
#[ORM\EntityListeners([CarrierListener::class])]
class UlozenkaOrder
{
	public const STATUS_COMPLETED = 'completed';
	public const STATUS_NOT_FOUND = 'notFound';

	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\Id]
	#[ORM\OneToOne(targetEntity: Order::class)]
	protected Order $order;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $parcelId = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $parcelName = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	private ?DateTime $exported = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	private ?string $exportType = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $numberPackage = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $lastStatus;

	#[ORM\Column(type: 'integer', options: ['unsigned' => true])]
	public int $serviceId;

	/** @var Collection<UlozenkaParcelNumber> */
	#[ORM\OneToMany(mappedBy: 'parcelOrder', targetEntity: UlozenkaParcelNumber::class, cascade: ['all'], orphanRemoval: true, indexBy: 'numberPackage')]
	public Collection $associatedNumberPackages;

	public function __construct(Order $order, int $serviceId)
	{
		$this->order                    = $order;
		$this->serviceId                = $serviceId;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function getParcelId(): ?string { return $this->parcelId; }

	public function export(string $type, ?DateTime $datetime = null): self
	{
		$this->exportType = $type;
		$this->exported   = $datetime ?: new DateTime();

		return $this;
	}

	public function getExportType(): ?string { return $this->exportType; }

	public function getExported(): ?DateTime { return $this->exported; }

	public function getTrackingUrl(): string
	{
		return sprintf((string) UlozenkaConfig::loadScalar('trackingUrl'), $this->numberPackage);
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf((string) UlozenkaConfig::loadScalar('trackingUrl'), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->lastStatus    = null;
		$this->exported      = null;
		$this->associatedNumberPackages->clear();
	}

	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(UlozenkaParcelNumber $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}
}
