<?php declare(strict_types = 1);

namespace Ulozenka\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use EshopOrders\Model\ExpeditionLogger;
use EshopOrders\Model\Helpers\OrderHelper;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Tracy\Debugger;
use Ulozenka\Model\Entities\UlozenkaOrder;
use UlozenkaLib\APIv3\Api;
use UlozenkaLib\APIv3\Enum\Attributes\LabelAttr;
use UlozenkaLib\APIv3\Enum\Endpoint;
use UlozenkaLib\APIv3\Model\Consignment\Address;
use UlozenkaLib\APIv3\Model\Consignment\Receiver;
use UlozenkaLib\APIv3\Resource\Consignments\Request\ConsignmentRequest;

class OrderApiService
{
	public static string $labelsDir = TMP_DIR . '/eshopOrders/ulozenkaLabels/';

	public function __construct(
		protected EntityManagerDecorator $em,
		protected ExpeditionLogger       $expeditionLogger,
	)
	{
	}

	public function generateOrderOverApi(array $orders, int $quantity = 1): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
		];
		$idShop = UlozenkaConfig::loadScalar('shopId');
		$apiKey = UlozenkaConfig::loadScalar('apiKey');

		foreach ($orders as $uo) {
			/** @var  UlozenkaOrder $uo */
			$order  = $uo->getOrder();
			$adress = $order->getAddressDelivery();

			$endpoint = Endpoint::PRODUCTION;

			$shopConfig = (array) UlozenkaConfig::load($order->site->getIdent()) ?: [];
			if ($shopConfig) {
				$idShop = $shopConfig['shopId'];
				$apiKey = $shopConfig['apiKey'];
			}

			$api = new Api($endpoint, $idShop, $apiKey);

			// Vytvoreni zakaznika
			$receiver = new Receiver;
			$receiver->setName($adress->getFirstName())
				->setSurname($adress->getLastName())
				->setPhone($adress->getPhone())
				->setEmail($adress->getEmail());
			if ($adress->getCompany()) {
				$receiver->setCompany($adress->getCompany());
			}

			// Pokud neni nastavena vydejna, nastavi se adresa zakaznika
			if (!$uo->parcelId) {
				$receiver->setAddress(new Address(
					$adress->getStreet(),
					$adress->getCity(),
					$adress->getPostal(),
				));
			}

			// Vytvoreni zasilky
			$consignmentRequest = new ConsignmentRequest($receiver, (string) $order->getId(), $quantity, $uo->serviceId);
			if ($uo->parcelId) {
				/** @phpstan-ignore-next-line */
				$consignmentRequest->setDestinationBranchId($uo->parcelId);
			}
			$consignmentRequest->setDestinationCountry($adress->getCountry()->getIso3166_1());

			$cod = 0;

			if ($order->getPaymentIdent() === 'cod') {
				$curr = $order->getCurrencyCode();
				$cod  = round($order->getPrice(true), $curr === 'CZK' ? 0 : $order->currency->decimals);

				if ($order->currency->decimals === 0) {
					$cod = (int) $cod;
				} else if ((!$adress && $curr === 'EUR')
					|| ($adress && Strings::lower($adress->getCountry()->getId()) === 'sk')) {
					$cod = OrderHelper::roundSkCod($cod);
				}
			}

			$consignmentRequest->setCashOnDelivery($cod);
			$consignmentRequest->setCurrency($order->getCurrencyCode());
			$consignmentRequest->setWeight((float) ($order->getItemsWeight() ?: 1));
			$consignmentRequest->setVariable((string) $order->getId());

			// Odeslani
			$createConsignmentRequest = $api->createConsignment($consignmentRequest);
			if ($createConsignmentRequest->isSuccess()) {
				$uo->export('api');
				$uo->numberPackage = (string) $createConsignmentRequest->getConsignment()->getId();
				$this->em->persist($uo);

				$spedition                 = $order->getSpedition();
				$spedition->trackingNumber = $uo->numberPackage;
				$spedition->trackingUrl    = $uo->getTrackingUrl();
				$this->em->persist($spedition);

				$this->em->flush();
				$result['ok']++;
			} else {
				$errors = [];
				foreach ($createConsignmentRequest->getErrors() as $error) {
					$this->expeditionLogger->logError('ulozenka', $error->getCode() . ' - ' . $error->getDescription(), $order->getId());

					$errors[] = $error->getCode() . ' - ' . $error->getDescription();
				}

				Debugger::log(json_encode(['order' => $order->getId(), 'errors' => $errors]), 'ulozenka-api-export');

				$result['error']++;
			}
		}

		return $result;
	}

	/**
	 * @param UlozenkaOrder[] $orders
	 *
	 * @throws Exception
	 */
	public function generateLabelPdf(array $orders, bool $sendToPrinter = false): array
	{
		$result   = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];
		$idShop   = UlozenkaConfig::loadScalar('shopId');
		$apiKey   = UlozenkaConfig::loadScalar('apiKey');
		$endpoint = Endpoint::PRODUCTION;

		$ordersByShop = [];
		foreach ($orders as $order) {
			$ordersByShop[$order->getOrder()->site->getIdent()][] = $order->numberPackage;
		}

		foreach ($ordersByShop as $shopIdent => $ids) {
			$shopConfig = (array) UlozenkaConfig::load($shopIdent, null) ?: [];
			if ($shopConfig) {
				$idShop = $shopConfig['shopId'];
				$apiKey = $shopConfig['apiKey'];
			}

			foreach ($ids as $k => $v) {
				$ids[$k] = (int) $v;
			}

			$api            = new Api($endpoint, $idShop, $apiKey);
			$labelsResponse = $api->getLabels($ids, $sendToPrinter ? LabelAttr::TYPE_ZPL : LabelAttr::TYPE_PDF, 1, 1, $idShop, $apiKey);

			if ($labelsResponse->isSuccess()) {
				$file = self::$labelsDir . time() . '_' . uniqid('', true) . '.pdf';
				FileSystem::createDir(dirname($file));
				$pdf = fopen($file, 'w');
				if (!$pdf) {
					throw new Exception('Cannot open file for writing: ' . $file);
				}

				fwrite($pdf, (string) $labelsResponse->getLabelsString());
				fclose($pdf);

				$result['files'][] = $file;
				$result['ok']++;
			} else {
				$result['error']++;

				$errors       = [];
				$firstOrderId = $ids[0];
				foreach ($labelsResponse->getErrors() as $error) {
					$this->expeditionLogger->logError('ulozenka', $error->getCode() . ' - ' . $error->getDescription(), (int) $firstOrderId);

					$errors[] = $error->getCode() . ' - ' . $error->getDescription();
				}

				Debugger::log(json_encode(['orders' => $ids, 'errors' => $errors]), 'ulozenka-api-export');
			}

			// Odstraneni starych stitku
			$today = (new DateTime())->format('Y-m-d');
			foreach (glob(self::$labelsDir . '*.pdf') ?: [] as $file) {
				$time = explode('_', basename($file))[0];

				if (Strings::isValidTimestamp((int) $time)) {
					$date = DateTime::from($time)->format('Y-m-d');

					if ($date < $today) {
						unlink($file);
					}
				}
			}
		}

		return $result;
	}

	public function checkCompleted(UlozenkaOrder $uo): bool
	{
		$shopIdent  = $uo->getOrder()->site->getIdent();
		$shopConfig = (array) UlozenkaConfig::load($shopIdent) ?: [];

		$idShop = (string) UlozenkaConfig::loadScalar('shopId');
		$apiKey = (string) UlozenkaConfig::loadScalar('apiKey');

		if ($shopConfig) {
			$idShop = $shopConfig['shopId'];
			$apiKey = $shopConfig['apiKey'];
		}

		$api = new Api(Endpoint::PRODUCTION, $idShop, $apiKey);

		$trackingResponse = $api->getTracking($uo->numberPackage);

		if ($trackingResponse->isSuccess()) {
			foreach ($trackingResponse->getStatuses() as $status) {
				if ($status->getId() === 10) {
					return true;
				}
			}
		} else {
			foreach ($trackingResponse->getErrors() as $error) {
				if ($error->getCode() == 4003) {
					$uo->lastStatus = UlozenkaOrder::STATUS_NOT_FOUND;
					$this->em->persist($uo);
					$this->em->flush();
					break;
				}

				Debugger::log('COMPLETE check error - ' . $error->getCode() . ' - ' . $error->getDescription(), 'ulozenka');
			}
		}

		return false;
	}
}
