<?php declare(strict_types = 1);

namespace Ulozenka\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\Helpers\BaseEntityService;
use Ulozenka\Model\Entities\UlozenkaOrder;

/**
 * @method UlozenkaOrder|null getReference($id)
 * @method UlozenkaOrder|null get($id)
 * @method UlozenkaOrder[] getAll()
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = UlozenkaOrder::class;

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return UlozenkaOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return UlozenkaOrder[]
	 */
	public function getOrdersExported(array $ids = [], bool $loadFull = true): array
	{
		return $this->getOrders(true, $ids, $loadFull);
	}

	/**
	 * @return UlozenkaOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->getEr()->createQueryBuilder('uo')
			         ->where('uo.lastStatus NOT IN (:status) OR uo.lastStatus IS NULL')
			         ->andWhere('uo.numberPackage IS NOT NULL')
			         ->setParameters(new ArrayCollection([new Parameter('status', [
					         UlozenkaOrder::STATUS_COMPLETED,
					         UlozenkaOrder::STATUS_NOT_FOUND,
				         ])]))->getQuery()
			         ->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return UlozenkaOrder[]
	 */
	public function getOrders(?bool $isExported = null, array $ids = [], bool $loadFull = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('uo')
			->addSelect('anp')
			->leftJoin('uo.associatedNumberPackages', 'anp')
			->orderBy('uo.order')->groupBy('uo.order');

		if ($loadFull) {
			$qb->addSelect('o, uo, oad, oai, os, s, oCurrency')
				->join('uo.order', 'o')
				->leftJoin('o.currency', 'oCurrency')
				->leftJoin('o.addressDelivery', 'oad')
				->leftJoin('o.addressInvoice', 'oai')
				->leftJoin('o.spedition', 'os')
				->leftJoin('os.spedition', 's');
		}

		if ($isExported === true) {
			$qb->where('uo.exported IS NOT NULL');
		} else if ($isExported === false) {
			$qb->where('uo.exported IS NULL');
		}

		if ($ids !== []) {
			$qb->andWhere('uo.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	public function markAsCompleted(array $orderIds): void
	{
		foreach ($this->getEr()->createQueryBuilder('uo')
			         ->where('uo.order IN (:ids)')
			         ->setParameter('ids', $orderIds)
			         ->getQuery()->getResult() as $row) {
			/** @var UlozenkaOrder $row */
			$row->lastStatus = UlozenkaOrder::STATUS_COMPLETED;
			$this->em->persist($row);
		}

		$this->em->flush();
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->getEr()->createQueryBuilder('u')
			         ->select('IDENTITY(u.order) as id')
			         ->leftJoin('u.associatedNumberPackages', 'acp')
			         ->orWhere('u.numberPackage LIKE :number')
			         ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
