<?php declare(strict_types = 1);

namespace Ulozenka\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Dao\PaymentSpedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Application\LinkGenerator;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Override;
use stdClass;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Ulozenka\Model\Entities\UlozenkaOrder;
use Ulozenka\Model\UlozenkaParcels;

class OrderFormSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected Request                $httpRequest,
		protected UlozenkaParcels        $ulozenkaParcelsService,
		protected Countries              $countriesService,
		protected PaymentSpeditions      $paymentSpeditions,
		protected LinkGenerator          $linkGenerator,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);

		$container = $form->addContainer('ulozenka');
		foreach ($this->getUlozenkaSpeditionIds() as $row) {
			if (!$row->getSpedition()->isPickup) {
				continue;
			}

			$speditionId = $row->getSpedition()->getId();

			foreach ($row->getCountries() as $country) {
				$k = $country->getId() . '_' . $speditionId;
				$container->addHidden($k);
				$container->addText($k . '_p')
					->setPlaceholder('ulozenka.chooseAddress')
					->setHtmlAttribute('readonly')
					->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('Ulozenka:Cron:Default:parcelsByService', [
						'id'      => $row->getSpedition()->code1,
						'country' => $country->getId(),
					]))
					->setHtmlAttribute('data-autocomplete-target', $container->getComponent($k)->getHtmlId());
			}
		}
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data = $event->data;
		$form = $event->form;
		/** @var ArrayHash $value */
		$value = $form->getUntrustedValues();

		foreach ($data['ulozenka'] ?? [] as $k => $v) {
			if (isset($form->getComponent('ulozenka', false)[$k])) {
				$form->getComponent('ulozenka')[$k]->setDefaultValue($v);
			}
		}

		$parcel = $this->getParcelByData($data);
		if ($parcel instanceof stdClass) {
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$form->getComponent($k . '2')->setValue($form->getComponent($k . '2')->isRequired() && !$value[$k] ? 'placeholder' : ($value[$k] ?: null));
			}

			$form->getComponent('useAddrDeli')->setValue(true);
			$form->getComponent('street2')->setValue(trim($parcel->street . ' ' . $parcel->house_number));
			$form->getComponent('city2')->setValue($parcel->town);
			$form->getComponent('postal2')->setValue(str_replace(' ', '', (string) $parcel->zip));
			$form->getComponent('country2')->setValue($data['speditionCountry']);
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$spedition          = $data['spedition'];
		$ulozenkaSpeditions = $this->getUlozenkaSpeditionIds();

		if (isset($ulozenkaSpeditions[$spedition])) {
			$spedition                   = $ulozenkaSpeditions[$spedition];
			$k                           = $data['speditionCountry'] . '_' . $data['spedition'];
			$data['ulozenka'][$k]        = $request->getPost('ulozenka')[$k];
			$data['ulozenka'][$k . '_p'] = $request->getPost('ulozenka')[$k . '_p'];
			if ($spedition->getSpedition()->isPickup) {
				$data['disableDeliveryAddressSpedition'] = 'ulozenka';
				$data['disableDeliveryAddress']          = true;
				$data['useAddrDeli']                     = false;
			} else if ($data['disableDeliveryAddressSpedition'] === 'ulozenka') {
				unset($data['disableDeliveryAddress']);
			}
		} else if ($data['disableDeliveryAddressSpedition'] === 'ulozenka') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		$parcel = $this->getParcelByData($data);
		if ($parcel instanceof stdClass) {
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$d[$k . '2'] = $data[$k];
			}

			$countryId   = $data['speditionCountry'];
			$countryText = $this->countriesService->getAllNameColumn()[$countryId] ?? $countryId;

			$d += [
				'useAddrDeli'  => true,
				'street2'      => trim($parcel->street . ' ' . $parcel->house_number),
				'city2'        => $parcel->town,
				'postal2'      => str_replace(' ', '', (string) $parcel->zip),
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form = $event->form;
		/** @var ArrayHash $values */
		$values = $form->getUntrustedValues();

		$spedition = $this->getUlozenkaSpeditionIds()[$values->speditions->{$values->speditionCountry}] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$parcelId = $values->ulozenka->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$parcelId) {
				$form->addError($this->translator->translate('ulozenka.orderForm.ulozenkaIdMissing'));
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values            = $event->formData;
		$ulozenkaSpedition = $this->getUlozenkaSpeditionIds();

		$spedition = $ulozenkaSpedition[$values['speditions'][$values['speditionCountry']]] ?? null;
		if ($spedition) {
			$spedition = $spedition->getSpedition();
			$entity    = new UlozenkaOrder($event->order, (int) $spedition->code1);

			if ($spedition->isPickup) {
				$parcelId = $values['ulozenka'][$values['speditionCountry'] . '_' . $spedition->getId()];
				$parcel   = $this->ulozenkaParcelsService->getParcelByService((int) $parcelId, (int) $spedition->code1);

				$entity->parcelId   = $parcelId;
				$entity->parcelName = $parcel->name;

				$event->order->getSpedition()->deliveryPointId = (string) $parcelId;
				$this->em->persist($event->order->getSpedition());
			}

			$this->em->persist($entity);
		}
	}

	/**
	 * @return PaymentSpedition[]
	 */
	protected function getUlozenkaSpeditionIds(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() !== 'ulozenka') {
				continue;
			}

			$result[$row->getSpedition()->getId()] = $row;
		}

		return $result;
	}

	public function getParcelByData(array $data): ?stdClass
	{
		$ulozenkaSpeditions = $this->getUlozenkaSpeditionIds();

		if (isset($ulozenkaSpeditions[$data['spedition']])) {
			$spedition = $ulozenkaSpeditions[$data['spedition']]->getSpedition();

			if ($spedition->isPickup) {
				$parcelId = $data['ulozenka'][$data['speditionCountry'] . '_' . $spedition->getId()] ?? null;

				if ($parcelId) {
					return $this->ulozenkaParcelsService->getParcelByService((int) $parcelId, (int) $spedition->code1);
				}
			}
		}

		return null;
	}
}
