<?php declare(strict_types = 1);

namespace Ulozenka\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\ControlEvent;
use Tracy\Debugger;
use Ulozenka\Model\Entities\UlozenkaOrder;
use Ulozenka\Model\Entities\UlozenkaParcelNumber;
use Ulozenka\Model\OrdersExported;
use Ulozenka\Model\UlozenkaParcels;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var UlozenkaOrder[]|null */
	protected static ?array $allOrders = null;
	protected OrdersExported $ordersExported;
	protected EntityManagerDecorator $em;
	protected UlozenkaParcels $ulozenkaParcels;
	protected EventDispatcher $eventDispatcher;

	public function __construct(OrdersExported  $ordersExported, EntityManagerDecorator $em, UlozenkaParcels $ulozenkaParcels,
	                            EventDispatcher $eventDispatcher)
	{
		$this->ordersExported  = $ordersExported;
		$this->em              = $em;
		$this->ulozenkaParcels = $ulozenkaParcels;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::createForm'    => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'      => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'ulozenka') {
			$control->template->ulozenkaExport              = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'ulozenka';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []), false);
			}

			$export = self::$allOrders[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrdersExported([$order->getId()]);
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$ulozenkaIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'ulozenka') {
				$ulozenkaIds[] = $sped->getId();
			}
		}

		if (empty($ulozenkaIds)) {
			return;
		}

		$container = $form->addContainer('ulozenka', '');
		$container->addText('parcelId', 'ulozenka.entity.parcelId')
			->setDescription('ulozenka.entity.parcelsList');
		$container->addText('numberPackage', 'ulozenka.entity.numberPackage')
			->setDescription('ulozenka.entity.numberPackageDesc');
		$container->addBool('removeExport', 'ulozenka.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'ulozenka.entity.updateDeliveryAddress')
			->setDefaultValue(1);

		$form->getComponent('spedition')->addCondition($form::IS_IN, $ulozenkaIds)
			->toggle($form['ulozenka']['parcelId']->getHtmlId())
			->toggle($form['ulozenka']['numberPackage']->getHtmlId())
			->toggle($form['ulozenka']['removeExport']->getHtmlId())
			->toggle($form['ulozenka']['updateDeliveryAddress']->getHtmlId());
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control       = $event->control;
		$orderId       = $orderSpedition->getOrder()->getId();
		$values        = $event->values;
		$spedition     = null;
		$ulozenkaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isUlozenka = false;
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'ulozenka' && $sped->getId() == $values->spedition) {
				$isUlozenka = true;
			}
			if ($sped->getId() == $values->spedition) {
				$spedition = $sped;
			}
		}


		try {
			if ($isUlozenka) {
				$parcel = null;
				if ($values->ulozenka->parcelId !== null && $values->ulozenka->parcelId !== '') {
					$parcel = $this->ulozenkaParcels->getParcelByService((int) $values->ulozenka->parcelId, (int) $spedition->code1);

					if (!$parcel) {
						$event->form->addError('ulozenka.orderForm.parcelNotFound');
						$control->redrawControl('form');

						return;
					}
				}

				if (!$ulozenkaOrder) {
					$ulozenkaOrder = new UlozenkaOrder($orderSpedition->getOrder(), (int) $spedition->code1);
				}

				$deliveryAddress = $ulozenkaOrder->getOrder()->getAddressDelivery();
				if ($deliveryAddress && $parcel && $values['ulozenka']->updateDeliveryAddress) {
					$deliveryAddress->setStreet(trim(sprintf('%s %s', $parcel->street, $parcel->house_number)));
					$deliveryAddress->setPostal($parcel->zip);
					$deliveryAddress->setCity($parcel->town);
					$this->em->persist($deliveryAddress);
					if ($event->presenter && isset($event->presenter['orderForm'])) {
						$event->presenter['orderForm']->redrawControl('addressDelivery');
					}
				}

				$numberPackages      = trim(str_replace(' ', '', (string) $values->ulozenka->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && Strings::contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);

				$ulozenkaOrder->parcelId = trim(str_replace(' ', '', (string) $values->ulozenka->parcelId)) ?: null;

				$orderSpedition->deliveryPointId = $ulozenkaOrder->parcelId ?: null;
				$this->em->persist($orderSpedition);

				if ($numberPackagesCount <= 1) {

					$ulozenkaOrder->numberPackage = $numberPackages[0] ?? null;
					$ulozenkaOrder->associatedNumberPackages->clear();
				} else {

					$existsPackages            = $ulozenkaOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$ulozenkaOrder->numberPackage = $numberPackage;
						} else {
							if (in_array($numberPackage, $diffToAdd)) {
								$ulozenkaOrder->associatedNumberPackages->add(new UlozenkaParcelNumber($numberPackage, $ulozenkaOrder));
							}
						}
					}

					foreach ($diffToRemove as $p) {
						$ulozenkaOrder->associatedNumberPackages->remove($p);
					}
				}

				if ($ulozenkaOrder->numberPackage && $ulozenkaOrder->getExported() === null) {
					$ulozenkaOrder->export('api');
				}

				if ($values->ulozenka->removeExport === 1) {
					$ulozenkaOrder->resetExport();
				}

				$this->em->persist($ulozenkaOrder);

				$this->eventDispatcher->dispatch(new Event(['order'          => $ulozenkaOrder->getOrder(),
				                                            'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($ulozenkaOrder) {
				$this->em->remove($ulozenkaOrder);
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('ulozenka.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$ulozenkaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$ulozenkaOrder) {
			return;
		}

		$event->form->getComponent('ulozenka')->setDefaults([
			'parcelId'      => $ulozenkaOrder->getParcelId(),
			'numberPackage' => implode(', ', $ulozenkaOrder->getAllNumberPackages()),
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'ulozenka')) {
			return;
		}

		$orderId       = $order->getId();
		$ulozenkaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$ulozenkaOrder) {
			return;
		}

		$ulozenkaOrder->resetExport();
		$this->em->persist($ulozenkaOrder);
		$this->em->flush($ulozenkaOrder);
	}
}
