<?php declare(strict_types = 1);

namespace Ulozenka\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\EshopOrdersConfig;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Ulozenka\Model\OrderApiService;
use Ulozenka\Model\OrdersExported;
use Ulozenka\Model\UlozenkaConfig;

class UlozenkaOrdersSubscriber implements EventSubscriberInterface
{
	protected OrdersExported  $ordersExported;
	protected OrderApiService $apiService;
	protected Translator      $translator;
	protected Speditions      $speditionsService;

	public function __construct(
		OrdersExported  $ordersExported,
		OrderApiService $apiService,
		Translator      $translator,
		Speditions      $speditionsService
	)
	{
		$this->ordersExported    = $ordersExported;
		$this->apiService        = $apiService;
		$this->translator        = $translator;
		$this->speditionsService = $speditionsService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => ['ordersGridAttached', 255],
		];
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (EshopOrdersConfig::load('hideOrderGridNavExport')) {
			return;
		}

		$event->addNavigation(
			'ulozenka.ordersNavigation.exportUlozenka',
			['navigationCallback!', ['key' => 'ulozenkaExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('ulozenka', 'ulozenka.ordersGrid.exportApi', function(array $ids, $quantity) use ($control) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);
			if (!$orders) {
				$control->presenter->flashMessageSuccess($this->translator->translate('ulozenka.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->generateOrderOverApi($orders, max(1, (int) $quantity));

			if ($result['error'] === 0) {
				$control->presenter->flashMessageSuccess($this->translator->translate('ulozenka.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter
					->flashMessageWarning($this->translator->translate('ulozenka.export.partial', null, $result));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('ulozenka.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		});

		if (UlozenkaConfig::load('allowGenerateLabel', false)) {
			$control->addExportCallback('ulozenkaLabels', 'ulozenka.ordersGrid.labels', function(array $ids) use ($control) {
				$orders = $this->ordersExported->getOrdersExported($ids);
				if (!$orders) {
					$control->presenter->flashMessageSuccess($this->translator->translate('ulozenka.labels.noOrders'));
					$control->presenter->redrawControl('flashes');

					return;
				}

				set_time_limit(120);
				$result = $this->apiService->generateLabelPdf($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$arr                                         = [
							'name' => 'Zásilkovna štítky',
							'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
								'file'     => $file,
								'filename' => basename($file),
							]))),
						];
						$control->presenter->payload->fileRequests[] = $arr;
					}

					$control->presenter->flashMessageSuccess($this->translator->translate('ulozenka.labels.success'));
				} else {
					$control->presenter->flashMessageDanger($this->translator->translate('ulozenka.labels.error'));
				}

				$control->presenter->redrawControl('flashes');
			});
		}

		if (UlozenkaConfig::load('allowGenerateZplPrint', false)) {
			$control->addExportCallback('ulozenkaZpl', 'ulozenka.ordersGrid.zplPrint', function(array $ids) use ($control) {
				$orders = $this->ordersExported->getOrdersExported($ids);
				if (!$orders) {
					$control->presenter->flashMessageSuccess($this->translator->translate('ulozenka.labels.noOrders'));
					$control->presenter->redrawControl('flashes');

					return;
				}

				set_time_limit(120);
				$result = $this->apiService->generateLabelPdf($orders, true);

				if ($result['error'] === 0) {
					$control->presenter
						->flashMessageSuccess($this->translator->translate('ulozenka.labels.success'));
				} else {
					$control->presenter->flashMessageDanger($this->translator->translate('ulozenka.labels.error'));
				}
				$control->presenter->redrawControl('flashes');
			});
		}
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'ulozenkaExport') {
			return;
		}

		$orders = $this->ordersExported->getOrdersNotExported();

		$ids = [];
		foreach ($orders as $k => $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if (!$orders) {
			$event->presenter->flashMessageSuccess($this->translator->translate('ulozenka.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->generateOrderOverApi($orders);

		if ($result['error'] === 0) {
			$event->presenter->flashMessageSuccess($this->translator->translate('ulozenka.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$event->presenter->flashMessageWarning($this->translator->translate('ulozenka.export.partial', null, $result));
		} else {
			$event->presenter->flashMessageDanger($this->translator->translate('ulozenka.export.error'));
		}

		$event->presenter->redrawControl('flashes');
	}
}
