<?php declare(strict_types = 1);

namespace Ulozenka\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Nette\Utils\DateTime;
use EshopOrders\Model\Entities\Order;
use Ulozenka\Model\UlozenkaConfig;

/**
 * @ORM\Table(name="ulozenka__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopOrders\Model\Listeners\CarrierListener"})
 */
class UlozenkaOrder
{
	public const STATUS_COMPLETED = 'completed';
	public const STATUS_NOT_FOUND = 'notFound';

	/**
	 * @ORM\Id
	 * @ORM\OneToOne(targetEntity="\EshopOrders\Model\Entities\Order")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Order $order;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $parcelId = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $parcelName = null;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private ?DateTimeInterface $exported = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	private ?string $exportType = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $numberPackage = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $lastStatus;

	/**
	 * @ORM\Column(type="integer", options={"unsigned": true})
	 */
	public int $serviceId;

	/**
	 * @var Collection<UlozenkaParcelNumber>
	 * @ORM\OneToMany(targetEntity="UlozenkaParcelNumber", mappedBy="parcelOrder", cascade={"all"}, orphanRemoval=true, indexBy="numberPackage")
	 */
	public Collection $associatedNumberPackages;

	public function __construct(Order $order, int $serviceId)
	{
		$this->order                    = $order;
		$this->serviceId                = $serviceId;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function getParcelId(): ?string { return $this->parcelId; }

	public function export(string $type, ?DateTime $datetime = null): self
	{
		$this->exportType = $type;
		$this->exported   = $datetime ?: new DateTime();

		return $this;
	}

	public function getExportType(): ?string { return $this->exportType; }

	public function getExported(): ?DateTimeInterface { return $this->exported; }

	public function getTrackingUrl(): string
	{
		return sprintf(UlozenkaConfig::load('trackingUrl', ''), $this->numberPackage);
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(UlozenkaConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->lastStatus    = null;
		$this->exported      = null;
		$this->associatedNumberPackages->clear();
	}

	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(UlozenkaParcelNumber $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}
}
