<?php declare(strict_types = 1);

namespace Ulozenka\Model;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseService;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Exception;
use Nette\Caching\Cache;
use Nette\Utils\Json;
use stdClass;
use Tracy\Debugger;

class UlozenkaParcels extends BaseService
{
	protected          $cache;
	public static ?int $overrideShopId = null;

	protected function getShopId(): int
	{
		$shopId = UlozenkaConfig::load('shopId');
		if (self::$overrideShopId) {
			$shopId = self::$overrideShopId;
		}

		return $shopId;
	}

	public function getParcelsByService(int $serviceId): array
	{
		$shopId = $this->getShopId();
		try {
			$list = $this->getCache()->load('service_' . $serviceId, function(&$dep) use ($serviceId, $shopId) {
				$dep = [Cache::EXPIRE => '1 hour'];

				$list = [];
				$json = file_get_contents('https://api.ulozenka.cz/v3/transportservices/' . $serviceId . '/branches?shopId=' . $shopId . '&includeInactive=0');
				foreach (Json::decode($json)->data->destination as $row) {
					$list[$row->id] = $row;
				}

				return $list;
			});
		} catch (Exception $e) {
			Debugger::log('ULOZENKA - cannot load parcels for service ' . $serviceId . '. ShopId:' . $shopId);

			return [];
		}

		return $list ?: [];
	}

	public function getBranch(int $branchId): ?stdClass
	{
		$shopId = $this->getShopId();
		try {
			$json = $this->getCache()->load('branch_' . $branchId, function(&$dep) use ($branchId, $shopId) {
				$dep = [Cache::EXPIRE => '1 hour'];

				return file_get_contents(sprintf('https://api.ulozenka.cz/v3/branches/%s?includeInactive=1&shopId=%s', $branchId, $shopId));
			});

			if (is_string($json) && Arrays::isJson($json)) {
				return Json::decode($json)->data[0];
			}

			LogNotifier::toDevelopers('Ulozenka parcel error ' . sprintf('https://api.ulozenka.cz/v3/branches/%s?includeInactive=1&shopId=%s', $branchId, $shopId), 'Ulozenka parcel error');

			return null;
		} catch (Exception $e) {
			LogNotifier::toDevelopers('ULOZENKA - cannot load branch ' . sprintf('https://api.ulozenka.cz/v3/branches/%s?includeInactive=1&shopId=%s', $branchId, $shopId), 'Ulozenka parcel error');
			Debugger::log($e->getMessage());
		}

		return null;
	}

	/**
	 * @return stdClass|null
	 */
	public function getParcelByService(int $parcelId, int $serviceId)
	{
		return $this->getParcelsByService($serviceId)[$parcelId] ?? null;
	}

	/**
	 * @return stdClass|null
	 */
	public function getParcelByShortcut(string $shortcut, int $serviceId)
	{
		foreach ($this->getParcelsByService($serviceId) as $row) {
			if ($row->shortcut === $shortcut || $row->id === (int) $shortcut) {
				return $row;
			}
		}

		return null;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, 'ulozenka');
		}

		return $this->cache;
	}
}
