<?php declare(strict_types = 1);

namespace Users\Model;

use Kdyby\Doctrine\EntityManager;
use Nette;
use Nette\Caching\Cache;
use Nette\Security\Permission;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;
use Users\Model\Entities\Role;

class Authorizator implements Nette\Security\IAuthorizator
{
	/** @var EntityManager */
	private $em;

	/** @var Cache */
	private $cache;

	/** @var Permission */
	private $acl;

	/** @var array */
	private $aclData = [];

	const CACHE_NAMESPACE = 'Users';

	public function __construct(EntityManager $em, Nette\Caching\IStorage $cacheStorage)
	{
		$this->em    = $em;
		$this->cache = new Cache($cacheStorage, self::CACHE_NAMESPACE);
		$this->acl   = new Permission();
	}

	public function setAcl($data)
	{
		$this->aclData = $data;

		$rolesKey = self::CACHE_NAMESPACE . '/roles';
		$roles    = $this->cache->load($rolesKey, function(&$dep) use ($rolesKey) {
			$dep = [Cache::TAGS => [self::CACHE_NAMESPACE, $rolesKey]];

			return $this->em->getRepository(Role::class)->findAll();
		});
		foreach ($roles as $role)
			$this->acl->addRole($role->name, $role->parent ? $role->parent->name : null);

		foreach (array_keys($data) as $resource)
			$this->acl->addResource($resource);

		$aclKey = self::CACHE_NAMESPACE . '/acl';
		$acl    = $this->cache->load($aclKey, function(&$dep) use ($aclKey) {
			$dep = [Cache::TAGS => [self::CACHE_NAMESPACE, $aclKey]];

			$acl        = $this->em->getRepository(Acl::class)->findAll();
			$privileges = $this->em->getRepository(Privilege::class)->findAll();
			$resources  = $this->em->getRepository(Resource::class)->findAll();

			return $acl;
		});
		foreach ($acl as $v) {
			$privilege = $v->privilege->name == 'all' ? Permission::ALL : $v->privilege->name;
			$resource  = $v->resource->name == 'all' ? Permission::ALL : $v->resource->name;

			try {
				$this->acl->{$v->allowed ? 'allow' : 'deny'}($v->role->name, $resource, $privilege);
			} catch (\Exception $e) {
			}
		}
	}

	/**
	 * @param $role
	 * @param $resource
	 * @param $privilege
	 *
	 * @return bool
	 */
	public function isAllowed($role, $resource, $privilege)
	{
		try {
			if (is_array($privilege)) {
				foreach ($privilege as $v)
					if ($this->acl->isAllowed($role, $resource, $v))
						return true;
				return false;
			} else
				return $this->acl->isAllowed($role, $resource, $privilege);
		} catch (Nette\InvalidStateException $e) {
			return false;
		}
	}

	public function getAclData() { return $this->aclData; }

	/**
	 * TODO pouze při aktualizaci verze CMS (porovnání obsahu souboru obsahující čas čas)
	 *
	 * @return bool
	 */
	public function updateDatabase()
	{
		$this->cache->clean([Cache::TAGS => self::CACHE_NAMESPACE]);

		$resources       = $this->em->getRepository(Resource::class)->findPairs([], 'name', [], 'id');
		$configResources = array_keys($this->aclData);
		$deleteResources = (array_diff($resources, $configResources));
		$addResources    = (array_diff($configResources, $resources));

		$privileges       = $this->em->getRepository(Privilege::class)->findPairs([], 'name', [], 'id');
		$configPrivileges = [];
		foreach ($this->aclData as $arr)
			foreach ($arr as $v)
				$configPrivileges[$v] = $v;
		$deletePrivileges = (array_diff($privileges, $configPrivileges));
		$addPrivileges    = (array_diff($configPrivileges, $privileges));

		if ($deletePrivileges || $deleteResources || $addPrivileges || $addResources) {
			$this->em->beginTransaction();
			try {
				if ($deleteResources)
					$this->em->createQuery("DELETE " . Resource::class . " r WHERE r.name IN (:names)")->setParameter('names', $deleteResources)->execute();
				if ($deletePrivileges)
					$this->em->createQuery("DELETE " . Privilege::class . " p WHERE p.name IN (:names)")->setParameter('names', $deletePrivileges)->execute();
				if ($addResources)
					foreach ($addResources as $resource)
						$this->em->persist(new Resource($resource));
				if ($addPrivileges)
					foreach ($addPrivileges as $privilege)
						$this->em->persist(new Privilege($privilege));

				$this->em->flush();
				$this->em->commit();
			} catch (\Exception $e) {
				$this->em->rollback();

				return false;
			}
		}

		return true;
	}
}