<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Nette\Application\ForbiddenRequestException;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;
use Users\Model\Roles;
use Users\Model\Security\IsRequired;

class UserForm extends BaseControl
{
	/** @var User */
	public $user;

	/** @var IsRequired */
	protected $isRequired;

	/** @var Roles */
	protected $rolesService;

	public function __construct(IsRequired $isRequired, Roles $roles)
	{
		$this->isRequired   = $isRequired;
		$this->rolesService = $roles;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		$roles = $this->rolesService->getEr()->findPairs([], 'name', ['name' => 'ASC']);

		$userRole = isset($this->getPresenter()->getUser()->getRoles()[0]) ? $this->getPresenter()->getUser()->getRoles()[0] : null;

		if ($userRole) {
			$remove = [1];
			$tmp    = $userRole;
			$i      = 0;
			while ($tmp->parent && $i < 100) {
				$remove[] = $tmp->parent->getId();
				$tmp      = $tmp->parent;
				$i++;
			}

			foreach ($roles as $k => $v) {
				if (in_array($k, $remove))
					unset($roles[$k]);
				elseif (!$this->isRequired->canSetRole($presenter->getUser(), $k))
					unset($roles[$k]);
			}
		} else {
			$roles = [];
		}

		$this['form']['roles']->setItems($roles);
		parent::attached($presenter);
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->getElementPrototype()->autocomplete = 'off';

		$form->addText('name', 'default.name')->setRequired();
		$form->addText('lastname', 'default.lastName')->setRequired();
		$form->addPassword('password', 'default.password')->setRequired();
		$form->addEmail('email', 'default.email')->setRequired();
		$form->addBool('isActive', 'default.isActive');
		$form->addRadioList('roles', 'default.roles', [])->setRequired();

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		try {
			$user = $this->user ?: new User($values->name, $values->email, $values->password);

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			$user->isActive = $values->isActive;

			if ($values->password != '')
				$user->setPassword($values->password);

			$roles = [$this->rolesService->getReference($values->roles)];
			$user->setRoles($roles);

			$this->em->persist($user)->flush();
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setUser($id)
	{
		$this->user = $this->em->getRepository(User::class)->find($id);

		$this->onAnchor[] = function() {
			$u    = $this->user;
			$form = $this['form'];
			$form->setDefaults([
				'name'     => $u->getName(),
				'lastname' => $this->user->getLastname(),
				'email'    => $u->getEmail(),
				'isActive' => $u->isActive(),
			]);

			if ($u->getRolesId() && array_key_exists($u->getRolesId()[0], $form['roles']->getItems()))
				$form['roles']->setDefaultValue($u->getRolesId()[0]);

			$form['password']->setRequired(false);

			if (!$this->isRequired->canEditUser($this->getPresenter()->getUser(), $this->user))
				throw new ForbiddenRequestException();
		};
	}
}
