<?php declare(strict_types = 1);

namespace Users\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\InvalidArgumentException;
use Nette\Security\IIdentity;
use Nette\Security\Passwords;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

/**
 * @ORM\Table(name="user")
 * @ORM\Entity
 * @ORM\EntityListeners({"AclListener"})
 */
class User implements IIdentity
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string", length=60)
	 */
	protected $name;

	/**
	 * @var string
	 * @ORM\Column(name="lastname", type="string", length=60)
	 */
	protected $lastname;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=60)
	 */
	protected $alias;

	/**
	 * @var string
	 * @ORM\Column(name="password", type="string", length=60)
	 */
	private $password;

	/**
	 * @var string
	 * @ORM\Column(name="email", type="string", length=60, nullable=true)
	 */
	protected $email;

	/**
	 * @var int
	 * @ORM\Column(name="is_active", type="smallint")
	 */
	protected $isActive;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var Role[]
	 * @ORM\ManyToMany(targetEntity="Role")
	 * @ORM\JoinTable(name="user_roles",
	 *     joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="role_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected $roles;
	
	/**
	 * @var UserAction
	 * @ORM\OneToMany(targetEntity="UserAction", mappedBy="user")
	 */
	protected $userActions;

	public function __construct($name, $email, $password)
	{
		$this->setName($name);
		$this->setEmail($email);
		$this->setPassword($password);
		$this->isActive = 1;
		$this->roles    = new ArrayCollection();
	}

	/*******
	 * === Name
	 */

	public function getName() { return $this->name; }

	public function setName($name)
	{
		$this->name  = $name;
		$this->alias = Strings::webalize($name);
	}

	/*******
	 * === LastName
	 */

	public function getLastname() { return $this->lastname; }

	public function setLastname($lastname)
	{
		$this->lastname = $lastname;
	}

	public function getAlias() { return $this->alias ?: Strings::webalize($this->name); }

	/*******
	 * === Password
	 */

	public function getPassword() { return $this->password; }

	public function setPassword($password)
	{
		$this->password = Passwords::hash($password);

		return $this;
	}

	public function isActive() { return $this->isActive; }

	public function disable($disabled = 1)
	{
		if (is_bool($disabled))
			$disabled = $disabled ? 1 : 0;

		$this->isActive = !$disabled;

		return $this;
	}

	public function getEmail() { return $this->email; }

	public function setEmail($email)
	{
		if ($email != '' && !Validators::isEmail($email))
			throw new InvalidArgumentException();
		$this->email = $email;

		return $this;
	}

	public function getCreated() { return $this->created; }

	public function getRoles() { return $this->roles; }

	public function getRolesId()
	{
		return $this->roles->map(function($role) { return $role->getId(); })->toArray();
	}

	public function getRolesString()
	{
		return implode(', ', $this->roles->map(function($role) { return $role->name; })->toArray());
	}

	public function setRoles($roles)
	{
		$this->roles = new ArrayCollection($roles);
	}

	public function addRole($role)
	{
		$this->roles->add($role);
	}

	public function isInRole($val)
	{
		foreach ($this->getRoles()->toArray() as $role) {
			if ($role->name == $val)
				return true;
		}

		return false;
	}
	
	public function isInRoleByIdent($ident)
	{
		foreach ($this->getRoles()->toArray() as $role) {
			if ($role->ident == $ident)
				return true;
		}
		
		return false;
	}
	
	public function getUserActions() { return $this->userActions; }
	
	/** @return UserAction|null */
	public function getUserActionByToken($token) {
		foreach ($this->userActions as $action) {
			if($action->getToken() == $token) {
				return $action;
			}
		}
		return false;
	}
	
	public function setUserActions($userActions)
	{
		$this->userActions = new ArrayCollection($userActions);
	}
	
	public function addUserAction($userAction)
	{
		$this->userActions->add($userAction);
	}
	
}
