<?php declare(strict_types = 1);

namespace Users\FrontModule\Components;

use Core\Model\UI\BaseControl;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Users\Model\BadPassParametersException;
use Users\Model\Entities\UserAction;
use Users\Model\Users;

class NewPasswordForm extends BaseControl
{
	/** @var Users */
	protected $usersService;

	public function __construct(Users $usersService)
	{
		$this->usersService = $usersService;
	}


	public function render()
	{
		$token = $this->getPresenter()->getParameter('token');
		$email = $this->getPresenter()->getParameter('email');
		if($this->checkValidParams($token, $email)) {
			$this->template->valid = true;
		} else {
			$this->template->valid = false;

			$link = $this->getPresenter()->link(':Users:Front:Login:forgotPassword');
			$this->template->forgotLink = $link;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addHidden('email')->setDefaultValue($this->getPresenter()->getParameter('email'));
		$form->addHidden('token')->setDefaultValue($this->getPresenter()->getParameter('token'));
		$form->addPassword('password', 'login.loginForm.password')->setRequired();
		$form->addPassword('password2', 'users.newPassword.confirmPassword')
			->addRule(Form::EQUAL, 'users.newPassword.errorEquals', $form['password'])->setRequired();

		$form->addSubmit('submit', 'users.newPassword.submit');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(Form $form)
	{
		$token = $form['token']->value;
		$email = $form['email']->value;
		if(!$this->checkValidParams($token, $email)) {
			$form->addError('users.newPassword.hashInvalid');
			$form->addError('users.newPassword.hashInvalidLink');
			$this->template->valid = false;

			$link = $this->getPresenter()->link(':Users:Front:Login:forgotPassword');
			$this->template->forgotLink = $link;
			$this->redrawControl('form');
		}
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		$presenter = $this->getPresenter();

		try {

			$user = $this->usersService->getByEmail($values->email);
			$user->setPassword($values->password);

			$tokenHash = md5($values->token);
			$action = $user->getUserActionByToken($tokenHash);
			$action->setDone(true);

			$this->em->persist($user);
			$this->em->persist($action);
			$this->em->flush();

		} catch (\Exception $e) {
			$form->addError('users.forgotPassword.error');

			if ($presenter->isAjax())
				$this->redrawControl('form');
			else
				return false;
		}

		$this->getPresenter()->flashMessageSuccess('users.newPassword.success');

		if ($this->getPresenter()->getUser()->isLoggedIn()) {
			$this->getPresenter()->getUser()->logout(true);
		}
		$this->getPresenter()->redirect(':Users:Front:Login:');
	}

	public function checkValidParams($token, $email)
	{
		if(!$email || !$token) {
			return false;
		}

		$user = $this->usersService->getByEmail($email);
		$tokenHash = md5($token);

		if(!$user) {
			return false;
		}

		$action = $user->getUserActionByToken($tokenHash);

		if(!$action || $action->isDone()) {
			return false;
		}

		$dateCreated = $action->getCreated();
		$dateLimit = new DateTime();
		$dateLimit->sub(new \DateInterval('P0DT24H'));
		if($dateCreated < $dateLimit) {
			return false;
		}

		if($action->getType() != UserAction::RESET_PASSWORD) {
			return false;
		}

		return true;
	}
}
