<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Helpers\BaseEntityService;
use Nette\Utils\Validators;
use Nette\Utils\FileSystem;
use Nette\Http\FileUpload;
use Users\Model\Entities\User;
use Users\Model\Entities\UserAction;

/**
 * Class Users
 * @package Users\Model
 *
 * @method User|null|object = getReference($id)
 * @method User[]|null getAll()
 * @method User|null get($id)
 */
class Users extends BaseEntityService
{
	protected $entityClass = User::class;

	/** @var UsersMailer */
	protected $usersMailer;

	public function __construct(UsersMailer $usersMailer)
	{
		$this->usersMailer = $usersMailer;
	}

	/** @return User|null */
	public function getByEmail($email): ?User
	{
		$query = $this->getEr()->createQueryBuilder('u', 'u.id');
		$query->andWhere('u.email = :email')->setParameter('email', $email);
		$user = $query->getQuery()->getOneOrNullResult();

		return $user;
	}

	/** Najde nebo vytvori uzivatele.
	 * Nejprve hleda prihlaseneho uzivatel. Kdyz neni, tak hleda podle emailu. Kdyz nenajde, vytvori uzivatele se zadanymi parametry
	 *
	 * @param int|null $userId
	 * @param string   $email
	 * @param string   $firstName
	 * @param string   $lastName
	 *
	 * @return User
	 */
	public function getOrCreateUser($userId, $email, $firstName = '', $lastName = ''): User
	{
		if ($userId) {
			$user = $this->get($userId);
		} else {
			$user = $this->getByEmail($email);
			if (!$user) {

				$randomPassword = md5(uniqid(random_bytes(5), true));
				$user           = new User($email, $randomPassword);
				$user->setName($firstName);
				$user->setLastname($lastName);
				$this->em->persist($user)->flush();
			}
		}

		return $user;
	}

	public function createForgotPasswordAction(string $email): bool
	{
		$user = $this->getByEmail($email);
		if (!$user)
			return false;

		try {
			$token      = md5(uniqid(random_bytes(5), true));
			$tokenHash  = md5($token);
			$userAction = new UserAction($user, UserAction::RESET_PASSWORD, $tokenHash);
			$user->addUserAction($userAction);
			$this->em->persist($userAction);
			$this->em->persist($user);
			$this->em->flush();
			$this->usersMailer->sendForgotPasswordLink($user->getEmail(), $token);
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function newUserEmailNotification(int $userId): bool
	{
		$user = $this->get($userId);
		if (!$user)
			return false;

		try {
			$token      = md5(uniqid(random_bytes(5), true));
			$tokenHash  = md5($token);
			$userAction = new UserAction($user, UserAction::RESET_PASSWORD, $tokenHash);
			$user->addUserAction($userAction);
			$this->em->persist($userAction);
			$this->em->persist($user)->flush($user);
			$this->usersMailer->sendNewUser($user, $token);
		} catch (\Exception $e) {

			return false;
		}

		return true;
	}

	/**
	 * @param FileUpload $fileUpload
	 * @param User       $user
	 * @return void
	 * @throws \Exception
	 */
	public function uploadProfileImage(FileUpload $fileUpload, User $user): void
	{
		$upload = null;

		try {
			$isNull = $fileUpload->getName() === null || $user->getProfileImage() === null;

			if (!$isNull && !$fileUpload->isOk()) {
				return;
			}

			$uploadsPath = $user->getUploadPath();

			if (!file_exists($uploadsPath)) {
				FileSystem::createDir($uploadsPath);
			}

			$dest = $user->getProfileImage();
			$fileUpload->move($dest);

			// to remove the file in case of error
			$upload = $dest;

		} catch (\Exception $e) {
			$tmpFile = $fileUpload->getTemporaryFile();
			if (!Validators::isNone($tmpFile) && file_exists($tmpFile)) {
				@unlink($tmpFile);
			}

			if ($upload !== null && file_exists($upload)) {
				@unlink($upload);
			}

			throw $e;
		}
	}

	/**
	 * @param int $userId
	 */
	public function removeProfileImage(int $userId): void
	{
		$user = $this->get($userId);

		if ($user === null) {
			return;
		}
		
		if ($user->getProfileImage() !== null) {
			FileSystem::delete($user->getProfileImage());
		}

		$user->setProfileImage(null);

		$this->em->persist($user)
		         ->flush($user);
	}
	
}
