<?php declare(strict_types = 1);

namespace Users\Model;

use Kdyby\Doctrine\EntityManager;
use Nette;
use Nette\Caching\Cache;
use Nette\Security\Permission;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;
use Users\Model\Entities\Role;

class Authorizator implements Nette\Security\IAuthorizator
{
	/** @var EntityManager */
	private $em;

	/** @var Cache */
	private $cache;

	/** @var Permission */
	private $acl;

	/** @var array */
	private $aclData = [];

	const CACHE_NAMESPACE = 'Users';

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function __construct(EntityManager $em, Nette\Caching\IStorage $cacheStorage)
	{
		$this->em    = $em;
		$this->cache = new Cache($cacheStorage, self::CACHE_NAMESPACE);
		$this->acl   = new Permission();
	}

	public function setAcl($data)
	{
		$this->aclData = $data;
		$key           = self::CACHE_NAMESPACE . '/roles-acl';

		$dbData = $this->cache->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			$roles = $this->em->getRepository(Role::class)->findAll();
			$this->em->getRepository(Privilege::class)->findAll();
			$this->em->getRepository(Resource::class)->findAll();
			$acl = $this->em->getRepository(Acl::class)->findAll();

			return ['roles' => $roles, 'acl' => $acl];
		});

		foreach ($dbData['roles'] as $role)
			$this->acl->addRole($role->ident, $role->parent ? $role->parent->ident : null);

		foreach (array_keys($data) as $resource)
			$this->acl->addResource($resource);

		foreach ($dbData['acl'] as $v) {
			$privilege = $v->privilege->name == 'all' ? Permission::ALL : $v->privilege->name;
			$resource  = $v->resource->name == 'all' ? Permission::ALL : $v->resource->name;

			try {
				$this->acl->{$v->allowed ? 'allow' : 'deny'}($v->role->ident, $resource, $privilege);
			} catch (\Exception $e) {
			}
		}
		$this->acl->allow('superAdmin');
	}

	/**
	 * @param $role
	 * @param $resource
	 * @param $privilege
	 *
	 * @return bool
	 */
	public function isAllowed($role, $resource = 'all', $privilege = 'all')
	{
		try {
			if (is_array($privilege)) {
				foreach ($privilege as $v)
					if ($this->acl->isAllowed($role, $resource, $v))
						return true;

				return false;
			} else
				return $this->acl->isAllowed($role, $resource, $privilege);
		} catch (Nette\InvalidStateException $e) {
			return false;
		}
	}

	public function getAclData() { return $this->aclData; }
}
