<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Sites;
use Nette\Localization\ITranslator;
use Users\Model\Entities\User;

class UsersMailer
{
	/** @var Sites */
	protected $sites;

	/** @var MailBuilderFactory */
	protected $mailer;

	/** @var ITranslator */
	protected $translator;

	public function __construct(Sites $sites, MailBuilderFactory $mailer, ITranslator $translator)
	{
		$this->sites      = $sites;
		$this->mailer     = $mailer;
		$this->translator = $translator;
	}

	public function sendNewUser(User $user, string $token): void
	{
		$mail = $this->mailer->create();

		$site    = $this->sites->getCurrentSite();
		$subject = $this->translator->translate('users.emails.createUser');

		$originTemplate = __DIR__ . '/../FrontModule/_emails/newUser.latte';
		$userTemplate   = TEMPLATES_DIR . '/Front/default/Users/_emails/newUser.latte';

		$mail->setSubject($subject);
		$mail->setParameters([
			'originTemplate' => $originTemplate,
			'subject'        => $subject,
			'email'          => $user->getEmail(),
			'token'          => $token,
		]);

		$mail->setTemplateFile(file_exists($userTemplate) ? $userTemplate : $originTemplate);
		$mail->setFrom($site->getEmail(), $site->getSiteName());
		$mail->addTo($user->getEmail());
		$mail->send();
	}

	public function sendForgotPasswordLink(string $email, string $token): void
	{
		$mail = $this->mailer->create();
		$site = $this->sites->getCurrentSite();

		$subject        = $this->translator->translate('usersFront.forgotPasswordForm.subject', ['siteName' => $site->getSiteName()]);
		$originTemplate = __DIR__ . '/../FrontModule/_emails/forgotPassword.latte';
		$userTemplate   = TEMPLATES_DIR . '/Front/default/Users/_emails/forgotPassword.latte';

		$mail->setSubject($subject);
		$mail->setParameters([
			'originTemplate' => $originTemplate,
			'subject'        => $subject,
			'email'          => $email,
			'token'          => $token,
		]);

		$mail->setTemplateFile(file_exists($userTemplate) ? $userTemplate : $originTemplate);
		$mail->setFrom($site->getEmail(), $site->getSiteName());
		$mail->addTo($email);
		$mail->send();
	}
}
