<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Helpers\BaseEntityService;
use Users\Model\Entities\User;
use Users\Model\Entities\UserAction;

/**
 * Class Users
 * @package Users\Model
 *
 * @method User|null|object = getReference($id)
 * @method User[]|null getAll()
 * @method User|null get($id)
 */
class Users extends BaseEntityService
{
	protected $entityClass = User::class;

	/** @var UsersMailer */
	protected $usersMailer;

	public function __construct(UsersMailer $usersMailer)
	{
		$this->usersMailer = $usersMailer;
	}

	/** @return User|null */
	public function getByEmail($email): ?User
	{
		$query = $this->getEr()->createQueryBuilder('u', 'u.id');
		$query->andWhere('u.email = :email')->setParameter('email', $email);
		$user = $query->getQuery()->getOneOrNullResult();

		return $user;
	}

	/** Najde nebo vytvori uzivatele.
	 * Nejprve hleda prihlaseneho uzivatel. Kdyz neni, tak hleda podle emailu. Kdyz nenajde, vytvori uzivatele se zadanymi parametry
	 *
	 * @param int|null $userId
	 * @param string   $email
	 * @param string   $firstName
	 * @param string   $lastName
	 *
	 * @return User
	 */
	public function getOrCreateUser($userId, $email, $firstName = '', $lastName = ''): User
	{
		if ($userId) {
			$user = $this->get($userId);
		} else {
			$user = $this->getByEmail($email);
			if (!$user) {

				$randomPassword = md5(uniqid(random_bytes(5), true));
				$user           = new User($email, $randomPassword);
				$user->setName($firstName);
				$user->setLastname($lastName);
				$this->em->persist($user)->flush();
			}
		}

		return $user;
	}

	public function createForgotPasswordAction(string $email): bool
	{
		$user = $this->getByEmail($email);
		if (!$user)
			return false;

		try {
			$token      = md5(uniqid(random_bytes(5), true));
			$tokenHash  = md5($token);
			$userAction = new UserAction($user, UserAction::RESET_PASSWORD, $tokenHash);
			$user->addUserAction($userAction);
			$this->em->persist($userAction);
			$this->em->persist($user);
			$this->em->flush();
			$this->usersMailer->sendForgotPasswordLink($user->getEmail(), $token);
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function newUserEmailNotification(int $userId): bool
	{
		$user = $this->get($userId);
		if (!$user)
			return false;

		try {
			$token      = md5(uniqid(random_bytes(5), true));
			$tokenHash  = md5($token);
			$userAction = new UserAction($user, UserAction::RESET_PASSWORD, $tokenHash);
			$user->addUserAction($userAction);
			$this->em->persist($userAction);
			$this->em->persist($user)->flush($user);
			$this->usersMailer->sendNewUser($user, $token);
		} catch (\Exception $e) {

			return false;
		}

		return true;
	}
}
