<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Users\Model\Authorizator;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;
use Users\Model\Entities\Role;

class RoleForm extends BaseControl
{
	/** @var Role */
	public $role;

	/** @var Authorizator */
	protected $authorizator;

	/** @var array */
	protected $aclConfig;

	protected $privileges;

	public function __construct($aclConfig, Authorizator $authorizator)
	{
		$this->aclConfig    = $aclConfig;
		$this->authorizator = $authorizator;
	}

	public function render()
	{
		if (!$this->getPresenter()->getUser()->isAllowed('Users:Admin', 'rolesManager'))
			return '';

		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
	}

	protected function createComponentForm()
	{
		$form             = $this->createForm();
		$resources        = $this->em->getRepository(Resource::class)->findPairs([], 'name', ['name' => 'ASC']);
		$this->privileges = $this->em->getRepository(Privilege::class)->findPairs([], 'name');

		$form->addText('name', 'users.roleForm.name')
			->setRequired();
		$form->addText('ident', 'users.roleForm.ident');

		$aclArr = [];
		foreach ($resources as $k => $resource) {
			$explResource = array_map(function($v) { return lcfirst($v); }, explode(':', $resource));
			$langDomain   = array_shift($explResource);

			if (empty($explResource))
				$explResource[] = 'all';

			if ($langDomain === 'all')
				$langDomain = 'admin';

			$langResources  = $langDomain . '.resources.';
			$langPrivileges = $langDomain . '.privileges.';
			$privileges     = $this->aclConfig[$resource];
			$items          = [];

			if (!$privileges)
				continue;

			foreach ($this->privileges as $pk => $pv) {
				if (in_array($pv, $privileges))
					$items[$pk] = $langPrivileges . $pv;
			}

			$aclArr[$k] = [
				'title' => $langResources . implode('_', $explResource),
				'items' => $items,
			];

			$form->addCheckboxList('r_' . $k, $resource, $items);
		}

		$form->onAnchor[] = function() use ($aclArr) {
			$this->template->aclArr = $aclArr;
		};

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			$role = $this->role ?: new Role($values->name);

			$role->name  = $values->name;
			$role->ident = Strings::webalize($values->ident ?: $values->name);

			$this->em->persist($role);
			$this->em->flush();

			$entityAcl = $this->em->getRepository(Acl::class)->findBy(['role' => $role->getId()]);

			$existsAcl = [];
			$formAcl   = [];

			foreach ($values as $k => $v) {
				if (strpos($k, 'r_') !== 0 || $v === false || empty($v))
					continue;

				$k = ltrim($k, 'r_');

				$formAcl[ltrim($k, 'r_')] = $v;
			}

			// Remove by resource
			foreach ($entityAcl as $k => $av) {
				if (!in_array($av->resource->getId(), array_keys($formAcl))) {
					$this->em->remove($av);
					unset($entityAcl[$k]);
				}

				$this->em->flush();
			}

			// Remove by privilege
			foreach ($entityAcl as $k => $av) {
				$fPrivileges = $formAcl[$av->resource->getId()];

				if (!in_array($av->privilege->getId(), $fPrivileges)) {
					$this->em->remove($av);
					unset($entityAcl[$k]);
				} else {
					$existsAcl[$av->resource->getId()][] = $av->privilege->getId();
				}

				$this->em->flush();
			}

			// Add acl
			foreach ($formAcl as $fResource => $fPrivileges) {
				foreach ($fPrivileges as $privilege) {
					if (isset($existsAcl[$fResource]) && in_array($privilege, $existsAcl[$fResource]))
						continue;

					$acl = new Acl(
						$role,
						$this->em->getReference(Resource::class, $fResource),
						$this->em->getReference(Privilege::class, $privilege)
					);

					$this->em->persist($acl);
				}

				$this->em->flush();
			}

			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	public function setRole($id)
	{
		$this->role = $this->em->getRepository(Role::class)->find($id);

		if ($this->role) {
			$this['form']->setDefaults([
				'name'  => $this->role->name,
				'ident' => $this->role->ident,
			]);

			$aclDefaults = [];
			foreach ($this->em->getRepository(Acl::class)->findBy(['role' => $this->role->getId()]) as $acl) {
				$resourceKey  = 'r_' . $acl->resource->getId();
				$privilegeKey = $acl->privilege->getId();
				$component    = $this['form']->getComponent($resourceKey, false);
				if (!$component || !isset($component->getItems()[$privilegeKey]))
					continue;

				$aclDefaults[$resourceKey][] = $privilegeKey;
			}

			$this['form']->setDefaults($aclDefaults);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
