<?php declare(strict_types = 1);

namespace Users\FrontModule\Components;

use Core\Model\Event\EventDispatcher;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;
use Users\Model\Event\UserEvent;
use Users\Model\Roles;
use Users\Model\Users;
use Users\Model\UsersConfig;

class RegisterForm extends BaseControl
{
	/** @var Users */
	protected $users;

	/** @var UsersConfig */
	protected $usersConfig;

	/** @var Roles */
	protected $roles;

	public function __construct(Users $users, EventDispatcher $eventDispatcher, UsersConfig $usersConfig, Roles $roles)
	{
		$this->users           = $users;
		$this->eventDispatcher = $eventDispatcher;
		$this->usersConfig     = $usersConfig;
		$this->roles           = $roles;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->getElementPrototype()->class[] = 'frm__w1';
		$form->setAjax();

		$form->addText('firstName', 'usersFront.registerForm.firstName')->setRequired();
		$form->addText('lastName', 'usersFront.registerForm.lastName')->setRequired();
		$form->addText('email', 'usersFront.registerForm.email')->setRequired();
		$form->addPassword('password', 'usersFront.registerForm.password')->setRequired();
		$form->addPassword('passwordCheck', 'usersFront.registerForm.passwordCheck')->setRequired()
			->addRule(BaseForm::EQUAL, 'usersFront.registerForm.passwordsDoNotMatch', $form['password']);

		if (SystemConfig::load('useGoogleRecaptcha', false) && UsersConfig::load('enableRegisterGoogleRecaptcha', true))
			$form->addInvisibleReCaptcha('recaptcha')->setRequired('default.formMessages.recaptcha');

		$form->addSubmit('submit', 'usersFront.registerForm.register')
			->getControlPrototype()->class[] = 'primary-bg-color';

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnValidate(BaseForm $form, ArrayHash $values): void
	{
		if ($this->users->getByEmail($values->email))
			$form->addError($this->t('usersFront.registerForm.emailExist'));

		if ($form->hasErrors())
			$this->redrawControl('form');
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->getPresenter();

		$this->em->beginTransaction();
		try {
			$user = new User($values->email, $values->password);
			$user->setName($values->firstName);
			$user->setLastname($values->lastName);

			$defaultRoles = $this->usersConfig->get('registerRoles');
			if ($defaultRoles) {
				foreach ($defaultRoles as $ident) {
					$role = $this->roles->getByIdent($ident);
					if ($role)
						$user->addRole($role);
				}
			}

			$event                     = new UserEvent($user);
			$event->data['formValues'] = $values;
			$this->eventDispatcher->dispatch($event, 'users.registerForm.beforeSave');

			$this->em->persist($user);
			$this->em->flush();
			$form->addCustomData('user', $user);
			$this->em->commit();
			$this->template->successMessage = $this->t('usersFront.registerForm.successMessage');
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();
			$form->addError($this->t('usersFront.registerForm.registerError'));
			$this->redrawControl('form');

			return false;
		}

		$event                     = new UserEvent($user);
		$event->data['formValues'] = $values;
		$this->eventDispatcher->dispatch($event, 'users.registerForm.afterSave');
		$presenter->redirect($this->usersConfig->get('redirectAfterRegister') ?: ':Users:Front:Login:');

		return true;
	}
}
