<?php declare(strict_types = 1);

namespace Users\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;
use Nette\InvalidArgumentException;
use Nette\Security\IIdentity;
use Nette\Security\Passwords;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

/**
 * @ORM\Table(name="user")
 * @ORM\Entity
 * @ORM\EntityListeners({"AclListener"})
 * @ORM\EntityListeners({"UserListener"})
 */
class User implements IIdentity
{
	use TId;

	/** @var string */
	protected const DEFAULT_BASE_DIR = 'users';

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string", length=60, nullable=true)
	 */
	public $name;

	/**
	 * @var string
	 * @ORM\Column(name="lastname", type="string", length=60, nullable=true)
	 */
	public $lastname;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=60, nullable=true)
	 */
	protected $alias;

	/**
	 * @var string
	 * @ORM\Column(name="password", type="string", length=60, nullable=false)
	 */
	private $password;

	/**
	 * @var string
	 * @ORM\Column(name="email", type="string", length=60, nullable=false)
	 */
	public $email;

	/**
	 * @var string|null
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public $profileImage;

	/**
	 * @var int
	 * @ORM\Column(name="is_active", type="smallint", nullable=false, options={"default": 1})
	 */
	public $isActive;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	protected $firstSignIn;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	protected $lastActivity;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $lastActivityInAdmin;

	/**
	 * @var Role[]
	 * @ORM\ManyToMany(targetEntity="Role", indexBy="id")
	 * @ORM\JoinTable(name="user_roles",
	 *     joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="role_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected $roles;

	/**
	 * @var UserAction
	 * @ORM\OneToMany(targetEntity="UserAction", mappedBy="user")
	 */
	protected $userActions;

	protected array $customData = [];

	public function __construct(string $email, string $password)
	{
		$this->setEmail($email);
		$this->setPassword($password);
		$this->isActive    = 1;
		$this->roles       = new ArrayCollection();
		$this->userActions = new ArrayCollection();
	}

	/*******
	 * === Name
	 */

	public function getName() { return $this->name; }

	public function setName($name)
	{
		$this->name  = $name;
		$this->alias = Strings::webalize($name);
	}

	/*******
	 * === LastName
	 */

	public function getLastname() { return $this->lastname; }

	public function setLastname($lastname)
	{
		$this->lastname = $lastname;
	}

	public function getAlias() { return $this->alias ?: Strings::webalize($this->name); }

	/*******
	 * === Password
	 */

	public function getPassword() { return $this->password; }

	public function setPassword($password)
	{
		$passwords      = new Passwords();
		$this->password = $passwords->hash($password);

		return $this;
	}

	public function isActive() { return $this->isActive; }

	public function disable($disabled = 1)
	{
		if (is_bool($disabled))
			$disabled = $disabled ? 1 : 0;

		$this->isActive = !$disabled;

		return $this;
	}

	public function getEmail() { return $this->email; }

	public function setEmail($email)
	{
		if ($email != '' && !Validators::isEmail($email))
			throw new InvalidArgumentException();
		$this->email = $email;

		return $this;
	}

	/**
	 * @return string
	 */
	private function getProfileImagesPath(): string
	{
		return UPLOADS_PATH . DS . self::DEFAULT_BASE_DIR . DS . 'profileImages';
	}

	/**
	 * @return string
	 */
	public function getProfileImagePath(): ?string
	{
		if ($this->profileImage === null) {
			return null;
		}

		return sprintf('%s/%s_%s', $this->getProfileImagesPath(), $this->getId(), $this->profileImage);
	}

	/**
	 * @return string
	 */
	public function getProfileImage(): ?string
	{
		if ($this->getProfileImagePath() === null) {
			return null;
		}

		return WWW_DIR . $this->getProfileImagePath();
	}

	/**
	 * @param string|null $profileImage
	 */
	public function setProfileImage(?string $profileImage): void
	{
		$this->profileImage = $profileImage;
	}

	/**
	 * @return string
	 */
	public function getUploadPath(): ?string
	{
		return sprintf('%s/%s', WWW_DIR, $this->getProfileImagesPath());
	}

	public function getCreated() { return $this->created; }

	public function getRoles(): array { return $this->roles->map(function($role) { return $role->ident; })->toArray(); }

	public function getRolesId() { return $this->roles->getKeys() ?: []; }

	public function getRolesString()
	{
		return implode(', ', $this->roles->map(function($role) { return $role->name; })->toArray());
	}

	public function setRoles($roles)
	{
		$this->roles = new ArrayCollection($roles);
	}

	public function addRole($role)
	{
		$this->roles->add($role);
	}

	public function isInRole($val)
	{
		return in_array($val, $this->getRoles());
	}

	public function isInRoleByIdent($ident)
	{
		return $this->isInRole($ident);
	}

	public function getRolesCollection() { return $this->roles; }

	public function getUserActions() { return $this->userActions; }

	/** @return UserAction|null */
	public function getUserActionByToken($token)
	{
		foreach ($this->userActions as $action) {
			if ($action->getToken() == $token) {
				return $action;
			}
		}

		return false;
	}

	public function setUserActions($userActions)
	{
		$this->userActions = new ArrayCollection($userActions);
	}

	public function addUserAction($userAction)
	{
		$this->userActions->add($userAction);
	}

	public function getFirstSignIn(): ?\DateTimeInterface { return $this->firstSignIn; }

	public function setFirstSignIn(DateTimeInterface $date): void
	{
		$this->firstSignIn = $date;
	}

	public function lastActivity() { return $this->lastActivity; }

	public function addCustomData(string $key, $value)
	{
		$this->customData[$key] = $value;
	}

	public function getCustomData(?string $key = null, $default = null)
	{
		if ($key)
			return $this->customData[$key] ?? $default;

		return $this->customData;
	}
}
