<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query\Expr\Join;
use Nette\Utils\Validators;
use Nette\Utils\FileSystem;
use Nette\Http\FileUpload;
use Users\Model\Entities\Acl;
use Users\Model\Entities\User;
use Users\Model\Entities\UserAction;

/**
 * Class Users
 * @package Users\Model
 *
 * @method User|null|object = getReference($id)
 * @method User[]|null getAll()
 */
class Users extends BaseEntityService
{
	protected $entityClass = User::class;

	protected array $cGet = [];

	public function __construct()
	{
	}

	/**
	 * @param $email
	 *
	 * @return User|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByEmail($email): ?User
	{
		$query = $this->getEr()->createQueryBuilder('u', 'u.id');
		$query->andWhere('u.email = :email')->setParameter('email', $email);
		$user = $query->getQuery()->setMaxResults(1)->getOneOrNullResult();

		return $user;
	}

	public function get($id): ?User
	{
		if (!isset($this->cGet[$id])) {
			$this->cGet[$id] = $this->getEr()->createQueryBuilder('u')
				->addSelect('r')
				->leftJoin('u.roles', 'r')
				->where('u.id = :id')->setParameter('id', $id)
				->getQuery()->getOneOrNullResult();
		}

		return $this->cGet[$id] ?? null;
	}

	/** Najde nebo vytvori uzivatele.
	 * Nejprve hleda prihlaseneho uzivatel. Kdyz neni, tak hleda podle emailu. Kdyz nenajde, vytvori uzivatele se zadanymi parametry
	 *
	 * @param int|null $userId
	 * @param string   $email
	 * @param string   $firstName
	 * @param string   $lastName
	 *
	 * @return User
	 */
	public function getOrCreateUser($userId, $email = null, $firstName = '', $lastName = ''): User
	{
		if ($userId) {
			$user = $this->get($userId);
		} else {
			$user = $this->getByEmail($email);
			if (!$user) {

				$randomPassword = md5(uniqid(random_bytes(5), true));
				$user           = new User($email, $randomPassword);
				$user->setName($firstName);
				$user->setLastname($lastName);
				$this->em->persist($user)->flush();
			}
		}

		return $user;
	}

	/**
	 * @param FileUpload $fileUpload
	 * @param User       $user
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function uploadProfileImage(FileUpload $fileUpload, User $user): void
	{
		$upload = null;

		try {
			$isNull = $fileUpload->getName() === null || $user->getProfileImage() === null;

			if (!$isNull && !$fileUpload->isOk()) {
				return;
			}

			$uploadsPath = $user->getUploadPath();

			if (!file_exists($uploadsPath)) {
				FileSystem::createDir($uploadsPath);
			}

			$dest = $user->getProfileImage();
			$fileUpload->move($dest);

			// to remove the file in case of error
			$upload = $dest;
		} catch (\Exception $e) {
			$tmpFile = $fileUpload->getTemporaryFile();
			if (!Validators::isNone($tmpFile) && file_exists($tmpFile)) {
				@unlink($tmpFile);
			}

			if ($upload !== null && file_exists($upload)) {
				@unlink($upload);
			}

			throw $e;
		}
	}

	/**
	 * @param int $userId
	 */
	public function removeProfileImage(int $userId): void
	{
		$user = $this->get($userId);

		if ($user === null) {
			return;
		}

		if ($user->getProfileImage() !== null) {
			FileSystem::delete($user->getProfileImage());
		}

		$user->setProfileImage(null);

		$this->em->persist($user)
			->flush($user);
	}

	/**
	 * @return User[]
	 */
	public function getUserMailWithLogPrivilege(): array
	{
		// roles
		$qb = $this->em->createQueryBuilder();
		$qb->select('r.id')
			->from(Acl::class, 'a')
			->join('a.privilege', 'p')
			->join('a.role', 'r', Join::WITH, 'p.name = \'logNotifications\'');
		$arr = array_map(static function(array $a) {
			return $a['id'];
		}, $qb->getQuery()->getResult(AbstractQuery::HYDRATE_ARRAY));

		// get users by selected roles
		$qb2 = $this->em->createQueryBuilder();
		$qb2->select('u')
			->from(User::class, 'u')
			->join('u.roles', 'r')
			->where($qb2->expr()->in('r.id', $arr));

		return array_map(static function(User $u) {
			return $u->getEmail();
		}, $qb2->getQuery()->getResult());
	}
}
