<?php declare(strict_types = 1);

namespace Users\FrontModule\Components;

use Nette\Application\UI\ITemplateFactory;
use Nette\Forms\Form;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Utils\ArrayHash;
use Core\Model\UI\BaseControl;
use Users\Model\Entities\UserAction;
use Users\Model\Navigation\ForgotPassword;
use Users\Model\Users;

class ForgotPasswordForm extends BaseControl
{
	const PAGE_PASSWORD_ID = 118;
	
	/** @var Users */
	protected $usersService;
	
	/** @var ITemplateFactory */
	protected $templateFactory;
	
	/** @var IMailer */
	protected $mailer;
	
	/** @var Message */
	protected $message;
	
	/** @var string */
	protected $siteName;
	
	public function __construct($data, string $siteName, Users $usersService, IMailer $mailer, ITemplateFactory $templateFactory)
	{
		$this->usersService = $usersService;
		
		$this->siteName					= $siteName;
		$this->message                  = new Message();
		$this->mailer                   = $mailer;
		$this->templateFactory          = $templateFactory;
		$this->message->setFrom($data['fromEmail'] ?? null, $data['fromName'] ?? null);
		$this->message->setSubject('');
	}
	
	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('email', 'login.loginForm.email')
			->addRule(Form::EMAIL)->setRequired();

		$form->addSubmit('submit', 'users.forgotPassword.submit');
		
		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}
	
	public function formOnValidate(Form $form)
	{
		$user = $this->usersService->getByEmail($form['email']->value);
		if(!$user) {
			$form->addError('users.forgotPassword.notExists');
			$this->redrawControl('form');
		}
	}
	
	public function formSuccess(Form $form, ArrayHash $values)
	{
		$presenter = $this->getPresenter();

		try {
			
			$token = md5(uniqid(random_bytes(5), true));
			$tokenHash = md5($token);
			
			$user = $this->usersService->getByEmail($values->email);
			$userAction = new UserAction($user, UserAction::RESET_PASSWORD, $tokenHash);
			$user->addUserAction($userAction);
			$this->em->persist($userAction);
			$this->em->persist($user);
			$this->em->flush();
			
			$this->sendEmailPassword($values->email, $token);
			
			$this->template->messages = array('users.forgotPassword.sent');
			$form->reset();
			$this->redrawControl('form');
		
		} catch (\Exception $e) {
			$form->addError('users.forgotPassword.error');

			if ($presenter->isAjax())
				$this->redrawControl('form');
			else
				return false;
		}
	}
	
	protected function sendEmailPassword($customerEmail, $passwordHash)
	{
		$subject = $this->translator->translate('users.forgotPassword.subject', ['siteName' => $this->siteName]);
		$this->message->setSubject($subject);
		
		$link = $this->getPresenter()->link(':Users:Front:Login:forgotPassword');
		$link .= '/' . ForgotPassword::URL_RESET;
		$link .= '?token=' . $passwordHash;
		$link .= '&email=' . $customerEmail;
		
		$template = $this->templateFactory->createTemplate();
		$template->setFile(__DIR__ . '/email.latte');
		$template->setParameters([
			'subject' => $this->message->getSubject(),
			'link' => $link,
		]);
		
		$this->message->setHtmlBody($template->renderToString());
		$this->message->addTo($customerEmail);
		
		$this->mailer->send($this->message);
	}
}
