<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Nette\Application\ForbiddenRequestException;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\Role;
use Users\Model\Entities\User;
use Users\Model\Roles;
use Users\Model\Security\IsRequired;

class UserForm extends BaseControl
{
	/** @var User */
	public $user;

	/** @var IsRequired */
	protected $isRequired;

	/** @var Roles */
	protected $rolesService;

	public function __construct(IsRequired $isRequired, Roles $roles)
	{
		$this->isRequired   = $isRequired;
		$this->rolesService = $roles;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		$roles     = $this->rolesService->getEr()->createQueryBuilder('r', 'r.id')
			->orderBy('r.name', 'ASC')->getQuery()->getResult();
		$userRoles = $this->getPresenter()->getUser()->getRoles();
		$remove    = [];
		if (!in_array(Role::SUPERADMIN, $userRoles)) {
			$remove[] = Role::SUPERADMIN;
			if (!in_array(Role::ADMIN, $userRoles))
				$remove[] = Role::ADMIN;
		}

		foreach ($roles as $k => $role) {
			if (in_array($role->ident, $remove)) {
				unset($roles[$k]);
				continue;
			}

			$tmp = $role->parent;
			$i   = 0;
			while ($tmp->parent && $i < 100) {
				unset($roles[$tmp->parent->getId()]);
				$tmp = $tmp->parent;
				$i++;
			}
		}

		foreach ($roles as $k => $v)
			$roles[$k] = $v->name;

		$this['form']['roles']->setItems($roles);
		parent::attached($presenter);
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->getElementPrototype()->autocomplete = 'off';

		$form->addText('name', 'default.name')->setRequired();
		$form->addText('lastname', 'default.lastName')->setRequired();
		$form->addPassword('password', 'default.password')->setRequired();
		$form->addEmail('email', 'default.email')->setRequired();
		$form->addBool('isActive', 'default.isActive');
		$form->addCheckboxList('roles', 'default.roles', [])->setRequired();

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		try {
			$user = $this->user ?: new User($values->email, $values->password);

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			$user->isActive = $values->isActive;

			if ($values->password != '')
				$user->setPassword($values->password);

			$userRoles = $user->getRolesId();
			$formRoles = $values->roles;

			$old = array_diff($userRoles, $formRoles);
			$new = array_diff($formRoles, $userRoles);

			foreach ($old as $id)
				$user->getRolesCollection()->remove($id);
			foreach ($new as $id)
				$user->getRolesCollection()->add($this->em->getReference(Role::class, $id));

			$this->em->persist($user)->flush();
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setUser($id)
	{
		$this->user = $this->em->getRepository(User::class)->find($id);

		$this->onAnchor[] = function() {
			$u    = $this->user;
			$form = $this['form'];
			$form->setDefaults([
				'name'     => $u->getName(),
				'lastname' => $this->user->getLastname(),
				'email'    => $u->getEmail(),
				'isActive' => $u->isActive(),
			]);

			$roles = [];
			foreach ($u->getRolesId() as $id) {
				if (array_key_exists($id, $form['roles']->getItems()))
					$roles[] = $id;
			}
			$form['roles']->setDefaultValue($roles);

			$form['password']->setRequired(false);

			if (!$this->isRequired->canEditUser($this->user))
				throw new ForbiddenRequestException();
		};
	}
}
