<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Users;

class NewPasswordForm extends BaseControl
{
	public function __construct(protected Users $usersService)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addHidden('email')
			->setDefaultValue($this->presenter->getParameter('email'));
		$form->addHidden('token')
			->setDefaultValue($this->presenter->getParameter('token'));
		$form->addPassword('password', 'login.newPasswordForm.password')
			->setAttribute('placeholder', 'login.newPasswordForm.password')
			->setRequired();
		$form->addPassword('password2', 'login.newPasswordForm.confirmPassword')
			->setAttribute('placeholder', 'login.newPasswordForm.confirmPassword')
			->addRule(Form::EQUAL, 'login.newPasswordForm.errorEquals', $form['password'])
			->setRequired();

		$form->addSubmit('submit', 'login.newPasswordForm.submit');

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(Form $form): void
	{
		$token = $form->getComponent('token')->value;
		$email = $form->getComponent('email')->value;
		if (!$this->checkValidParams($token, $email)) {
			$link = $this->presenter->link(':Users:Admin:Login:forgotPassword');
			$form->addError($this->t('login.newPasswordForm.hashInvalid', null, ['link' => $link]));
			$this->redrawControl('form');
		}
	}

	public function formSuccess(Form $form, ArrayHash $values): void
	{
		try {
			$user = $this->usersService->getByEmail($values->email);
			$user->setPassword($values->password);

			$tokenHash = md5((string) $values->token);
			$action    = $user->getUserActionByToken($tokenHash);
			$action->setDone(true);

			$this->em->persist($user);
			$this->em->persist($action);
			$this->em->flush();
			$form->reset();

			$this->flashMessage('login.newPasswordForm.success');
		} catch (Exception) {
			$form->addError('login.newPasswordForm.error');
		}

		$this->redrawControl('form');
	}

	public function checkValidParams(?string $token, ?string $email): bool
	{
		if (!$email || !$token) {
			return false;
		}

		return $this->usersService->checkUserToken($token, $email);
	}

}
