<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\AbortException;
use Nette\Application\BadRequestException;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;
use Users\Model\Entities\User;
use Users\Model\Users;

class ProfileForm extends BaseControl
{
	public ?User $user = null;

	public function __construct(protected Users $users)
	{
	}

	public function render(): void
	{
		if (!$this->user) {
			return;
		}

		$this->template->userProfile = $this->user;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'default.name')->setRequired();
		$form->addText('lastname', 'default.lastName')->setRequired();
		$form->addPassword('password', 'default.password')->setRequired();
		$form->addText('phone', 'default.phone');
		$form->addEmail('email', 'default.email')->setRequired();

		$form->addFilesManager('profileImage', 'users.profileForm.profileImage');

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values): void
	{
		try {
			$user = $this->user ?: new User($values->email, $values->password);

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			$user->setProfileImage($values->profileImage);
			$user->phone = $values->phone;

			if (!Validators::isNone($values->password)) {
				$user->setPassword($values->password);
			}

			$this->presenter->flashMessageSuccess('users.profileForm.profileUpdated');
			$this->em->persist($user)->flush();
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}
	}

	/**
	 * @throws BadRequestException
	 */
	public function setUser(int|string $id): void
	{
		$this->user = $this->em->getRepository(User::class)->find($id);

		if ($this->user) {
			$this['form']->setDefaults([
				'name'         => $this->user->getName(),
				'lastname'     => $this->user->getLastname(),
				'email'        => $this->user->getEmail(),
				'profileImage' => $this->user->profileImage,
				'phone'        => $this->user->phone,
			]);

			$this['form']->getComponent('password')->setRequired(false);
		} else {
			$this->presenter->error();
		}
	}

	/**
	 * @throws AbortException
	 */
	public function handleRemoveProfileImage(int|string $id): never
	{
		$this->users->removeProfileImage((int) $id);
		$this->presenter->redirect('this');
	}
}
