<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\BadRequestException;
use Nette\Caching\Cache;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Authorizator;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;
use Users\Model\Entities\Role;

class RoleForm extends BaseControl
{
	public ?Role $role = null;

	/** @var string[] */
	protected array $privileges;

	public function __construct(protected array $aclConfig, protected Authorizator $authorizator)
	{
	}

	public function render(): void
	{
		if (!$this->presenter->getUser()->isAllowed('Users:Admin', 'rolesManager')) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form             = $this->createForm();
		$resources        = $this->em->getRepository(Resource::class)->findPairs([], 'name', ['name' => 'ASC']);
		$this->privileges = $this->em->getRepository(Privilege::class)->findPairs([], 'name');

		$form->addText('name', 'users.roleForm.name')
			->setRequired();
		$form->addText('ident', 'users.roleForm.ident');

		$aclArr = [];
		foreach ($resources as $k => $resource) {
			$explResource = array_map(fn($v) => lcfirst((string) $v), explode(':', (string) $resource));
			$langDomain   = array_shift($explResource);

			if (empty($explResource)) {
				$explResource[] = 'all';
			}

			if ($langDomain === 'all') {
				$langDomain = 'admin';
			}

			$langResources  = $langDomain . '.resources.';
			$langPrivileges = $langDomain . '.privileges.';
			$privileges     = $this->aclConfig[$resource];
			$items          = [];

			if (!$privileges) {
				continue;
			}

			foreach ($this->privileges as $pk => $pv) {
				if (in_array($pv, $privileges)) {
					$items[$pk] = $langPrivileges . $pv;
				}
			}

			$aclArr[$k] = [
				'title' => $langResources . implode('_', $explResource),
				'items' => $items,
			];

			$form->addCheckboxList('r_' . $k, $resource, $items);
		}

		$form->onAnchor[] = function() use ($aclArr) {
			$this->template->aclArr = $aclArr;
		};

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$role = $this->role ?: new Role($values->name);

			$role->name  = $values->name;
			$role->ident = str_replace(['-', ' '], ['',
				''], Strings::toAscii((string) ($values->ident ?: $values->name)));

			$this->em->persist($role);
			$this->em->flush();

			$entityAcl = $this->em->getRepository(Acl::class)->findBy(['role' => $role->getId()]);

			$existsAcl = [];
			$formAcl   = [];

			foreach ($values as $k => $v) {
				if (!str_starts_with($k, 'r_') || $v === false || empty($v)) {
					continue;
				}

				$k = ltrim($k, 'r_');

				$formAcl[ltrim($k, 'r_')] = $v;
			}

			// Remove by resource
			foreach ($entityAcl as $k => $av) {
				if (!in_array($av->resource->getId(), array_keys($formAcl))) {
					$this->em->remove($av);
					unset($entityAcl[$k]);
				}

				$this->em->flush();
			}

			// Remove by privilege
			foreach ($entityAcl as $k => $av) {
				$fPrivileges = $formAcl[$av->resource->getId()];

				if (!in_array($av->privilege->getId(), $fPrivileges)) {
					$this->em->remove($av);
					unset($entityAcl[$k]);
				} else {
					$existsAcl[$av->resource->getId()][] = $av->privilege->getId();
				}

				$this->em->flush();
			}

			// Add acl
			foreach ($formAcl as $fResource => $fPrivileges) {
				foreach ($fPrivileges as $privilege) {
					if (isset($existsAcl[$fResource]) && in_array($privilege, $existsAcl[$fResource])) {
						continue;
					}

					/** @var Privilege $privilegeEntity */
					$privilegeEntity = $this->em->getReference(Privilege::class, $privilege);

					/** @var Resource $resourceEntity */
					$resourceEntity = $this->em->getReference(Resource::class, $fResource);

					$acl = new Acl(
						$role,
						$resourceEntity,
						$privilegeEntity
					);

					$this->em->persist($acl);
				}

				$this->em->flush();
			}

			$this->em->commit();

			$cache = new Cache($this->cacheStorage, Authorizator::CACHE_NAMESPACE);
			$cache->remove(Authorizator::CACHE_NAMESPACE . '/roles-acl');

			return true;
		} catch (Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
		}

		return false;
	}

	/**
	 * @throws BadRequestException
	 */
	public function setRole(string|int $id): void
	{
		$this->role = $this->em->getRepository(Role::class)->find($id);

		if ($this->role) {
			$this['form']->setDefaults([
				'name'  => $this->role->name,
				'ident' => $this->role->ident,
			]);

			$aclDefaults = [];
			foreach ($this->em->getRepository(Acl::class)->findBy(['role' => $this->role->getId()]) as $acl) {
				$resourceKey  = 'r_' . $acl->resource->getId();
				$privilegeKey = $acl->privilege->getId();
				$component    = $this['form']->getComponent($resourceKey, false);
				if (!$component || !isset($component->getItems()[$privilegeKey])) {
					continue;
				}

				$aclDefaults[$resourceKey][] = $privilegeKey;
			}

			$this['form']->setDefaults($aclDefaults);
		} else {
			$this->presenter->error();
		}
	}
}
