<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Users\Model\Entities\Role;
use Users\Model\Roles;

class RolesGrid extends BaseControl
{
	public function __construct(protected Roles $rolesService)
	{
	}

	public function render(): void
	{
		if (!$this->presenter->getUser()->isAllowed('Users:Admin', 'rolesManager')) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->rolesService->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->em->getRepository(Role::class)->createQueryBuilder('u');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('name', 'users.rolesGrid.name', 'Roles:edit');
		$grid->addColumntext('ident', 'users.rolesGrid.ident');

		// Actions
		$grid->addAction('edit', '', 'Roles:edit')->setIcon('edit')->setBsType('primary')
			->setRenderCondition(function(Role $row) {
				if ($row->ident === Role::SUPERADMIN && !in_array(
						Role::SUPERADMIN,
						$this->presenter->getUser()
							->getRoles(),
					)) {
					return false;
				}

				return true;
			});
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setRenderCondition(fn(Role $row) => $row->ident !== Role::SUPERADMIN)
			->setConfirm('default.reallyDelete');

		return $grid;
	}
}
