<?php declare(strict_types = 1);

namespace Users\AdminModule\Components;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\ForbiddenRequestException;
use Nette\Application\IPresenter;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;
use Users\Model\Authorizator;
use Users\Model\Entities\Role;
use Users\Model\Entities\User;
use Users\Model\Roles;
use Users\Model\Security\IsRequired;
use Users\Model\Users;
use Users\Model\UsersConfig;

class UserForm extends BaseControl
{
	public ?User $user = null;

	public function __construct(protected IsRequired $isRequired, protected Roles $rolesService, protected Users $users)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param IComponent $presenter
	 */
	protected function attached($presenter): void
	{
		$user   = $this->presenter->getUser();
		$roles  = $this->rolesService->getEr()->createQueryBuilder('r', 'r.id')
			->orderBy('r.name', 'ASC')->getQuery()->getResult();
		$remove = [];

		if (!$user->isInRole(Role::SUPERADMIN)) {
			$remove[] = Role::SUPERADMIN;
			if (!$user->isAllowed('Users:Admin', 'adminManager')) {
				$remove[] = Role::ADMIN;
			}
		}

		foreach ($roles as $k => $role) {
			if (in_array($role->ident, $remove)) {
				unset($roles[$k]);
				continue;
			}

			$tmp = $role->parent;
			$i   = 0;
			while ($tmp->parent && $i < 100) {
				unset($roles[$tmp->parent->getId()]);
				$tmp = $tmp->parent;
				$i++;
			}
		}

		foreach ($roles as $k => $v) {
			$roles[$k] = $v->name;
		}

		$this['form']->getComponent('roles')->setItems($roles);

		if ($this->user) {
			$dRoles = [];
			foreach ($this->user->getRolesId() as $id) {
				if (isset($roles[$id])) {
					$dRoles[] = $id;
				}
			}

			$this['form']->getComponent('roles')->setDefaultValue($dRoles);
		}

		parent::attached($presenter);
	}

	public function handleLoadUsers(): never
	{
		$output = $this->users->getAllNonAdminUsersBasicData();

		$this->presenter->sendJson($output);
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->getElementPrototype()->autocomplete = 'off';

		$form->addText('name', 'default.name')->setRequired();
		$form->addText('lastname', 'default.lastName')->setRequired();
		$form->addEmail('email', 'default.email')
			->setHtmlAttribute('autocomplete', 'false')
			->setRequired();
		$form->addPassword('password', 'default.password')
			->setHtmlAttribute('autocomplete', 'false')
			->setRequired();
		$form->addText('phone', 'default.phone');
		$form->addFilesManager('profileImage', 'users.profileForm.profileImage');
		$form->addBool('isActive', 'default.isActive')
			->setDefaultValue(1);
		$form->addCheckboxList('roles', 'default.roles', []);

		if (UsersConfig::load('allowLoginAsAnotherUser')) {
			$form->addHidden('loginAs');
			$form->addText('loginAsPlaceholder', 'users.user.loginAsAnotherUser')
				->setHtmlAttribute('data-autocomplete-name', 'usersLoginAsAnotherUser')
				->setHtmlAttribute('data-autocomplete-target', $form->getComponent('loginAs')->getHtmlId())
				->setHtmlAttribute('data-autocomplete-keys', 'name');

			$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
				$form->getComponent('loginAsPlaceholder')
					->setHtmlAttribute('data-autocomplete-url', $this->link('loadUsers!'));
			});
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			self::class . '::createForm',
		);

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = $this->formValidate(...);
		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;
		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $this->template, $presenter),
			self::class . '::validateForm',
		);
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$user = $this->user ?: new User($values->email, $values->password);

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			$user->profileImage = $values->profileImage;
			$user->isActive     = $values->isActive;
			$user->phone        = $values->phone;

			if (UsersConfig::load('allowLoginAsAnotherUser') && is_numeric($values->loginAs)) {
				if ($values->loginAsPlaceholder !== '' && isset(
						$this->users->getAllNonAdminUsersBasicData()[$values->loginAs],
					)) {
					/** @var User $userEntity */
					$userEntity = $this->em->getReference(User::class, $values->loginAs);

					$user->loginAs = $userEntity;
				} else {
					$user->loginAs = null;
				}
			}

			if (!Validators::isNone($values->password)) {
				$user->setPassword($values->password);
			}

			$userRoles = $user->getRolesId();
			$formRoles = $values->roles;

			$old = array_diff($userRoles, $formRoles);
			$new = array_diff($formRoles, $userRoles);

			foreach ($old as $id) {
				$user->getRolesCollection()->remove($id);
			}
			foreach ($new as $id) {
				$user->getRolesCollection()->add($this->em->getReference(Role::class, $id));
			}

			$presenter               = $this->presenter;
			$event                   = new FormSuccessEvent($form, $values, $this->template, $presenter);
			$event->custom['entity'] = &$user;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

			$this->em->persist($user);
			$this->em->flush();

			$cache = new Cache($this->cacheStorage, Authorizator::CACHE_NAMESPACE);
			$cache->remove(Authorizator::CACHE_NAMESPACE . '/roles-acl');

			return true;
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setUser(string|int $id): void
	{
		$this->user = $this->em->getRepository(User::class)->find($id);

		$this->onAnchor[] = function() {
			$u = $this->user;
			/** @var BaseForm $form */
			$form = $this['form'];
			$d    = [
				'name'         => $u->getName(),
				'lastname'     => $this->user->getLastname(),
				'email'        => $u->getEmail(),
				'isActive'     => $u->isActive(),
				'profileImage' => $u->profileImage,
				'phone'        => $u->phone,
			];

			if (UsersConfig::load('allowLoginAsAnotherUser') && $u->loginAs) {
				$d['loginAs']            = $u->loginAs->getId();
				$d['loginAsPlaceholder'] = $u->loginAs->getId() . ' | ' . $u->loginAs->getFullName(
					) . ' (' . $u->loginAs->getEmail() . ')';
			}

			$form->setDefaults($d);

			$form->getComponent('password')->setRequired(false);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->user), self::class . '::setUser');

			if (!$this->isRequired->canEditUser($this->user)) {
				throw new ForbiddenRequestException();
			}
		};
	}
}
