<?php declare(strict_types = 1);

namespace Users\FrontModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Users;
use Users\Model\UsersConfig;

class NewPasswordForm extends BaseControl
{
	public function __construct(protected Users $usersService)
	{
	}

	public function render(): void
	{
		$token = $this->presenter->getParameter('token');
		$email = $this->presenter->getParameter('email');
		if ($this->checkValidParams($token, $email)) {
			$this->template->valid = true;
		} else {
			$this->template->valid = false;

			$link                       = $this->presenter->link(':Users:Front:Login:forgotPassword');
			$this->template->forgotLink = $link;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addHidden('email')->setDefaultValue($this->presenter->getParameter('email'));
		$form->addHidden('token')->setDefaultValue($this->presenter->getParameter('token'));
		$form->addPassword('password', 'usersFront.newPasswordForm.password')->setRequired();
		$form->addPassword('password2', 'usersFront.newPasswordForm.confirmPassword')
			->addRule(Form::EQUAL, 'usersFront.newPasswordForm.errorEquals', $form['password'])->setRequired();

		$form->addSubmit('submit', 'usersFront.newPasswordForm.submit');

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(Form $form): void
	{
		$token = $form->getComponent('token')->value;
		$email = $form->getComponent('email')->value;
		if (!$this->checkValidParams($token, $email)) {
			$form->addError('usersFront.newPasswordForm.hashInvalid');
			$form->addError('usersFront.newPasswordForm.hashInvalidLink');
			$this->template->valid = false;

			$link                       = $this->presenter->link(':Users:Front:Login:forgotPassword');
			$this->template->forgotLink = $link;
			$this->redrawControl('form');
		}
	}

	public function formSuccess(Form $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;

		try {

			$user = $this->usersService->getByEmail($values->email);
			$user->setPassword($values->password);

			$tokenHash = md5((string) $values->token);
			$action    = $user->getUserActionByToken($tokenHash);
			$action->setDone(true);

			$this->em->persist($user);
			$this->em->persist($action);
			$this->em->flush();
		} catch (Exception) {
			$form->addError('usersFront.newPasswordForm.error');

			if ($presenter->isAjax()) {
				$this->redrawControl('form');
			} else {
				return;
			}
		}

		$this->presenter->flashMessageSuccess('usersFront.newPasswordForm.success');
		$presenter->getUser()->login($values->email, $values->password);

		$presenter->redirect(UsersConfig::load('redirectAfterLogin'));
	}

	public function checkValidParams(?string $token, ?string $email): bool
	{
		if (!$email || !$token) {
			return false;
		}

		return $this->usersService->checkUserToken($token, $email);
	}
}
