<?php declare(strict_types = 1);

namespace Users\FrontModule\Components;

use Core\Model\Event\EventDispatcher;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;
use Users\Model\Event\UserEvent;
use Users\Model\Roles;
use Users\Model\Users;
use Users\Model\UsersConfig;

class RegisterForm extends BaseControl
{
	public function __construct(
		protected Users       $users,
		EventDispatcher       $eventDispatcher,
		protected UsersConfig $usersConfig,
		protected Roles       $roles,
	)
	{
		$this->eventDispatcher = $eventDispatcher;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->getElementPrototype()->class[] = 'frm__w1';
		$form->setAjax();

		$form->addText('firstName', 'usersFront.registerForm.firstName')->setRequired();
		$form->addText('lastName', 'usersFront.registerForm.lastName')->setRequired();
		$form->addText('email', 'usersFront.registerForm.email')->setRequired();
		$form->addPassword('password', 'usersFront.registerForm.password')->setRequired();
		$form->addPassword('passwordCheck', 'usersFront.registerForm.passwordCheck')->setRequired()
			->addRule($form::EQUAL, 'usersFront.registerForm.passwordsDoNotMatch', $form['password']);

		if (UsersConfig::load('registerForm.showNewsletterSubscriber')) {
			$form->addCheckbox('newsletterSubscribe', 'usersFront.registerForm.newsletterSubscribe');
		}

		if (SystemConfig::load('useGoogleRecaptcha', false) && UsersConfig::load(
				'enableRegisterGoogleRecaptcha',
				true,
			)) {
			if (SystemConfig::load('googleRecaptchaInvisible') === true) {
				$form->addInvisibleReCaptcha('recaptcha', true, 'usersFront.registerForm.recaptcha')
					->setRequired('default.formMessages.recaptchaRequired');
			} else {
				$form->addReCaptcha(
					'recaptcha',
					'usersFront.registerForm.recaptcha',
					true,
					'default.formMessages.recaptcha',
				)
					->setRequired('default.formMessages.recaptchaRequired');
			}
		}

		$form->addSubmit('submit', 'usersFront.registerForm.register')
			->getControlPrototype()->class[] = 'primary-bg-color';

		$form->onValidate[] = $this->formOnValidate(...);
		$form->onSuccess[]  = $this->formOnSuccess(...);

		return $form;
	}

	public function formOnValidate(BaseForm $form, ArrayHash $values): void
	{
		if ($this->users->getByEmail($values->email)) {
			$form->addError($this->t('usersFront.registerForm.emailExist'));
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;

		$this->em->beginTransaction();
		try {
			$user = new User($values->email, $values->password);
			$user->setName($values->firstName);
			$user->setLastname($values->lastName);

			$defaultRoles = $this->usersConfig->get('registerRoles');
			if ($defaultRoles) {
				foreach ($defaultRoles as $ident) {
					$role = $this->roles->getByIdent($ident);
					if ($role) {
						$user->addRole($role);
					}
				}
			}

			$event                     = new UserEvent($user);
			$event->data['formValues'] = $values;
			$this->eventDispatcher->dispatch($event, 'users.registerForm.beforeSave');

			$this->em->persist($user);
			$this->em->flush();
			$form->addCustomData('user', $user);
			$this->em->commit();
			$this->presenter->flashMessageSuccess('usersFront.registerForm.successMessage');
			$this->template->successMessage = $this->t('usersFront.registerForm.successMessage');
		} catch (Exception) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$form->addError($this->t('usersFront.registerForm.registerError'));
			$this->redrawControl('form');
		}

		if (isset($user)) {
			$event                     = new UserEvent($user);
			$event->data['formValues'] = $values;
			$this->eventDispatcher->dispatch($event, 'users.registerForm.afterSave');
		}

		$presenter->redirect($this->usersConfig->get('redirectAfterRegister') ?: ':Users:Front:Login:');
	}
}
