<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Nette;
use Nette\Security\AuthenticationException;
use Nette\Security\IIdentity;
use Nette\Security\Passwords;
use Users\Model\Entities\User;
use Users\Model\Security\FakeIdentity;

class Authenticator implements Nette\Security\Authenticator, Nette\Security\IdentityHandler
{

	/** @var string */
	protected          $identityClass = User::class;
	protected string   $userType      = 'user';
	public static bool $loginByEmail  = false;

	public function __construct(protected EntityManagerDecorator $em, protected Users $users)
	{
	}

	public function authenticate(string $username, string $password): IIdentity
	{
		$user = $this->findUser($username);

		if ($user && self::$loginByEmail) {
			return new FakeIdentity($user->getId(), User::class);
		}

		$passwords = new Passwords;

		if (!$user) {
			throw new AuthenticationException('login.messages.badLogin', self::IDENTITY_NOT_FOUND);
		} else if (!$passwords->verify($password, $user->getPassword())) {
			throw new AuthenticationException('login.messages.badLogin', self::INVALID_CREDENTIAL);
		} else if ($passwords->needsRehash($user->getPassword())) {
			$user->setPassword($passwords->hash($password));
			$this->em->persist($user);
			$this->em->flush($user);
		} else if ($user->isActive()) {
			$loggedByAnotherUser = null;
			if (UsersConfig::load('allowLoginAsAnotherUser') && $user->loginAs
				&& isset($this->users->getAllNonAdminUsersBasicData()[$user->loginAs->getId()])) {
				$loggedByAnotherUser = $user->getId();
				$user                = $user->loginAs;
			}

			$identity = new FakeIdentity($user->getId(), $this->identityClass, $this->userType);
			if ($loggedByAnotherUser) {
				$identity->loggedByAnotherUser = $loggedByAnotherUser;
			}

			$identity = $user;

			return $identity;
		}

		throw new AuthenticationException('login.messages.badLogin', self::IDENTITY_NOT_FOUND);
	}

	protected function findUser(string $username): ?User
	{
		return $this->em->getRepository(User::class)->findOneBy(['email' => $username, 'isActive' => 1]);
	}

	public function sleepIdentity(IIdentity $identity): IIdentity
	{
		// zde lze pozměnit identitu před zápisem do úložiště po přihlášení,
		// ale to nyní nepotřebujeme
		return $identity;
	}

	public function wakeupIdentity(IIdentity $identity): ?IIdentity
	{
		// aktualizace rolí v identitě
		$userId = $identity->getId();

		if ($userId) {
			return $this->users->get($userId);
		}

		return $identity;
	}
}
