<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Exception;
use Nette;
use Nette\Caching\Cache;
use Nette\Security\Permission;
use Throwable;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;
use Users\Model\Entities\Role;

class Authorizator implements Nette\Security\Authorizator
{
	protected readonly Cache      $cache;
	protected readonly Permission $acl;
	private array                 $aclData = [];
	final public const CACHE_NAMESPACE = 'Users';

	protected array $cacheDep = [
		Cache::Tags    => [self::CACHE_NAMESPACE],
		Cache::Expire  => '1 week',
		Cache::Sliding => true,
	];

	public function __construct(protected readonly EntityManagerDecorator $em, Nette\Caching\IStorage $cacheStorage)
	{
		$this->cache = new Cache($cacheStorage, self::CACHE_NAMESPACE);
		$this->acl   = new Permission();
	}

	/**
	 * @param array[] $data
	 *
	 * @throws Throwable
	 */
	public function setAcl(array $data): void
	{
		$this->aclData = $data;
		$key           = self::CACHE_NAMESPACE . '/roles-acl';

		$dbData = $this->cache->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			$roles = $this->em->getRepository(Role::class)->findAll();
			$this->em->getRepository(Privilege::class)->findAll();
			$this->em->getRepository(Resource::class)->findAll();
			$acl = $this->em->getRepository(Acl::class)->findAll();

			return ['roles' => $roles, 'acl' => $acl];
		});

		foreach ($dbData['roles'] as $role) {
			$this->acl->addRole($role->ident, $role->parent ? $role->parent->ident : null);
		}

		foreach (array_keys($data) as $resource) {
			$this->acl->addResource($resource);
		}

		foreach ($dbData['acl'] as $v) {
			$privilege = $v->privilege->name == 'all' ? Permission::ALL : $v->privilege->name;
			$resource  = $v->resource->name == 'all' ? Permission::ALL : $v->resource->name;

			try {
				$this->acl->{$v->allowed ? 'allow' : 'deny'}($v->role->ident, $resource, $privilege);
			} catch (Exception) {
			}
		}
		$this->acl->allow('superAdmin');
	}

	/**
	 * @param string|null       $role
	 * @param string|null       $resource
	 * @param array|string|null $privilege
	 */
	public function isAllowed($role, $resource = 'all', $privilege = 'all'): bool
	{
		try {
			if (is_array($privilege)) {
				foreach ($privilege as $v) {
					if ($this->acl->isAllowed($role, $resource, $v)) {
						return true;
					}
				}

				return false;
			}

			return $this->acl->isAllowed($role, $resource, $privilege);
		} catch (Nette\InvalidStateException) {
		}

		return false;
	}

	public function getAclData(): array { return $this->aclData; }
}
