<?php declare(strict_types = 1);

namespace Users\Model\Entities;

use Core\Model\Entities\TId;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\InvalidArgumentException;
use Nette\Security\IIdentity;
use Nette\Security\Passwords;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Users\Model\Entities\Repository\UserRepository;
use Users\Model\Listeners\AclListener;
use Users\Model\Listeners\UserListener;

#[ORM\Table(name: 'user')]
#[ORM\Entity]
#[ORM\EntityListeners([AclListener::class, UserListener::class])]
class User implements IIdentity
{
	use TId;

	protected const DEFAULT_BASE_DIR = 'users';

	#[ORM\Column(name: 'name', type: Types::STRING, length: 60, nullable: true)]
	public ?string $name;

	#[ORM\Column(name: 'lastname', type: Types::STRING, length: 60, nullable: true)]
	public ?string $lastname;

	#[ORM\Column(name: 'alias', type: Types::STRING, length: 60, nullable: true)]
	protected ?string $alias;

	#[ORM\Column(name: 'password', type: Types::STRING, length: 60, nullable: false)]
	private string $password;

	#[ORM\Column(name: 'email', type: Types::STRING, length: 60, nullable: false)]
	public string $email;

	#[ORM\Column(name: 'phone', type: Types::STRING, length: 60, nullable: true)]
	public ?string $phone = null;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $profileImage;

	#[ORM\Column(name: 'is_active', type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $isActive = 1;

	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(name: 'created', type: Types::DATETIME_MUTABLE)]
	protected ?DateTimeInterface $created = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $firstSignIn = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $lastActivity = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTimeInterface $lastActivityInAdmin = null;

	/**
	 * @var Collection<Role>
	 */
	#[ORM\JoinTable(name: 'user_roles')]
	#[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'role_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Role::class, indexBy: 'id')]
	protected Collection $roles;

	/**
	 * @var Collection<UserAction>
	 */
	#[ORM\OneToMany(mappedBy: 'user', targetEntity: UserAction::class)]
	protected Collection $userActions;

	#[ORM\ManyToOne(targetEntity: User::class)]
	#[ORM\JoinColumn(name: 'login_as', referencedColumnName: 'id', onDelete: 'SET NULL')]
	public ?self $loginAs = null;

	#[ORM\Column(name: 'last_modified', type: Types::DATETIME_MUTABLE, nullable: false, options: ['default' => 'CURRENT_TIMESTAMP'])]
	public ?DateTimeInterface $lastModified = null;

	#[ORM\Column(name: 'params', type: Types::ARRAY, nullable: true)]
	private ?array $params = [];

	protected array $customData = [];

	public function __construct(string $email, string $password)
	{
		$this->setEmail($email);
		$this->setPassword($password);
		$this->isActive     = 1;
		$this->roles        = new ArrayCollection();
		$this->userActions  = new ArrayCollection();
		$this->lastModified = new DateTime();
		$this->params       = [];
	}

	public function getFullName(): string { return trim($this->getName() . ' ' . $this->getLastname()); }

	/*******
	 * === Name
	 */

	public function getName(): ?string { return $this->name; }

	public function setName(string $name): void
	{
		$this->name  = $name;
		$this->alias = Strings::webalize($name);
	}

	/*******
	 * === LastName
	 */

	public function getLastname(): ?string { return $this->lastname; }

	public function setLastname(?string $lastname): void
	{
		$this->lastname = $lastname;
	}

	public function getAlias(): ?string { return $this->alias ?: Strings::webalize($this->name); }

	/*******
	 * === Password
	 */

	public function getPassword(): string { return $this->password; }

	public function setPassword(string $password): self
	{
		$passwords      = new Passwords;
		$this->password = $passwords->hash($password);

		return $this;
	}

	/**
	 * @return int
	 */
	public function isActive() { return $this->isActive; }

	/**
	 * @param bool|int<0, 1> $disabled
	 */
	public function disable($disabled = 1): self
	{
		if (is_bool($disabled)) {
			$disabled = $disabled ? 1 : 0;
		}

		$this->isActive = (int) !$disabled;

		return $this;
	}

	public function getEmail(): string { return $this->email; }

	public function setEmail(string $email): self
	{
		if (!Validators::isEmail($email)) {
			throw new InvalidArgumentException;
		}
		$this->email = $email;

		return $this;
	}

	private function getProfileImagesPath(): string
	{
		return WWW_DIR . DS . 'images';
	}

	public function getProfileImagePath(): ?string
	{
		return $this->profileImage;
	}

	public function getProfileImage(): ?string
	{
		if ($this->getProfileImagePath() === null) {
			return null;
		}

		return WWW_DIR . $this->getProfileImagePath();
	}

	public function setProfileImage(?string $profileImage): void
	{
		$this->profileImage = $profileImage;
	}

	public function getUploadPath(): ?string
	{
		return sprintf('%s/%s', WWW_DIR, $this->getProfileImagesPath());
	}

	public function getCreated(): ?DateTimeInterface { return $this->created; }

	/**
	 * @return string[]
	 */
	public function getRoles(): array { return $this->roles->map(function($role) { return $role->ident; })->toArray(); }

	/**
	 * @return array|int[]|string[]|Role[]
	 */
	public function getRolesId() { return $this->roles->getKeys() ?: []; }

	public function getRolesString(): string
	{
		return implode(', ', $this->roles->map(function($role) { return $role->name; })->toArray());
	}

	public function setRoles(array $roles): void
	{
		$this->roles = new ArrayCollection($roles);
	}

	public function addRole(Role $role): void
	{
		$this->roles->add($role);
	}

	public function isInRole(string $val): bool
	{
		return in_array($val, $this->getRoles(), true);
	}

	public function isInRoleByIdent(string $ident): bool
	{
		return $this->isInRole($ident);
	}

	public function getRolesCollection(): Collection { return $this->roles; }

	public function getUserActions(): Collection { return $this->userActions; }

	public function getUserActionByToken(string $token): ?UserAction
	{
		foreach ($this->userActions as $action) {
			if ($action->getToken() === $token) {
				return $action;
			}
		}

		return null;
	}

	/**
	 * @param UserAction[] $userActions
	 */
	public function setUserActions(array $userActions): void
	{
		$this->userActions = new ArrayCollection($userActions);
	}

	public function addUserAction(UserAction $userAction): void
	{
		$this->userActions->add($userAction);
	}

	public function getFirstSignIn(): ?DateTimeInterface { return $this->firstSignIn; }

	public function setFirstSignIn(DateTimeInterface $date = new DateTime()): void
	{
		$this->firstSignIn = $date;
	}

	public function lastActivity(): ?DateTimeInterface { return $this->lastActivity; }

	/**
	 * @param mixed $value
	 */
	public function addCustomData(string $key, $value): void
	{
		$this->customData[$key] = $value;
	}

	/**
	 * @param mixed $default
	 *
	 * @return mixed
	 */
	public function getCustomData(?string $key = null, $default = null)
	{
		if ($key) {
			return $this->customData[$key] ?? $default;
		}

		return $this->customData;
	}

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value = null): void
	{
		if (!is_array($this->params)) {
			$this->params = [];
		}

		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}
	}

	/**
	 * @return mixed|null
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

}
