<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImageHelper;
use DateInterval;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use Exception;
use Nette\Http\FileUpload;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Validators;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Role;
use Users\Model\Entities\User;
use Users\Model\Entities\UserAction;
use Users\Model\Event\UserEvent;

/**
 * @method User|null|object getReference($id)
 * @method User[]|null getAll()
 */
class Users extends BaseEntityService
{
	protected $entityClass = User::class;

	protected array  $cGet              = [];
	protected ?array $cAllNonAdminUsers = null;
	protected ?array $cAllAdmins        = null;

	public function __construct(protected EventDispatcher $eventDispatcher)
	{
	}

	public function getByEmail(string $email): ?User
	{
		$query = $this->getEr()->createQueryBuilder('u', 'u.id');
		$query->andWhere('u.email = :email')->setParameter('email', $email);

		return $query->getQuery()->setMaxResults(1)->getOneOrNullResult();
	}

	/**
	 * @param int|string|int[]|string[] $id
	 *
	 * @throws NonUniqueResultException
	 */
	public function get($id): ?User
	{
		if (is_array($id)) {
			$id = array_values($id)[0];
		}

		if (!isset($this->cGet[$id])) {
			$this->cGet[$id] = $this->getEr()->createQueryBuilder('u')
				->addSelect('r')
				->leftJoin('u.roles', 'r')
				->where('u.id = :id')->setParameter('id', $id)
				->getQuery()->getOneOrNullResult();
		}

		return $this->cGet[$id] ?? null;
	}

	/**
	 * Najde nebo vytvori uzivatele.
	 * Nejprve hleda prihlaseneho uzivatel. Kdyz neni, tak hleda podle emailu. Kdyz nenajde, vytvori uzivatele se zadanymi parametry
	 *
	 * @param string $email
	 * @param string $firstName
	 * @param string $lastName
	 *
	 * @throws Exception
	 */
	public function getOrCreateUser(?int $userId, $email = null, $firstName = '', $lastName = ''): User
	{
		if ($userId) {
			$user = $this->get($userId);
		} else {
			$user = $this->getByEmail($email);

			if (!$user) {
				$randomPassword = md5(uniqid(random_bytes(5), true));
				$user           = new User($email, $randomPassword);
				$user->setName($firstName);
				$user->setLastname($lastName);
				$this->em->persist($user);
				$this->em->flush();

				$event = new UserEvent($user);
				$this->eventDispatcher->dispatch($event, 'users.userCreated');
			}
		}

		return $user;
	}

	public function uploadProfileImage(FileUpload $fileUpload, User $user): void
	{
		/** @var string|null $upload */
		$upload = null;

		try {
			$isNull = !$fileUpload->getName() || !$user->getProfileImage();

			if (!$isNull && !$fileUpload->isOk()) {
				return;
			}

			$uploadsPath = $user->getUploadPath();

			if (!file_exists($uploadsPath)) {
				FileSystem::createDir($uploadsPath);
			}

			$dest = $user->getProfileImage();
			$fileUpload->move($dest);
			ImageHelper::autoResize($dest);

			// to remove the file in case of error
			$upload = $dest;
		} catch (Exception $e) {
			$tmpFile = $fileUpload->getTemporaryFile();
			if (!Validators::isNone($tmpFile) && file_exists($tmpFile)) {
				@unlink($tmpFile);
			}

			if ($upload !== null && file_exists($upload)) {
				@unlink($upload);
			}

			throw $e;
		}
	}

	public function removeProfileImage(int $userId): void
	{
		$user = $this->get($userId);

		if ($user === null) {
			return;
		}

		if ($user->getProfileImage() !== null) {
			FileSystem::delete($user->getProfileImage());
		}

		$user->setProfileImage(null);

		$this->em->persist($user)
			->flush($user);
	}

	/**
	 * @return string[]
	 */
	public function getUserMailWithLogPrivilege(): array
	{
		// roles
		$qb = $this->em->createQueryBuilder();
		$qb->select('r.id')
			->from(Acl::class, 'a')
			->join('a.privilege', 'p')
			->join('a.role', 'r', Join::WITH, 'p.name = \'logNotifications\'');
		$arr = array_map(static fn(array $a) => $a['id'], $qb->getQuery()->getResult(AbstractQuery::HYDRATE_ARRAY));

		// get users by selected roles
		$qb2 = $this->em->createQueryBuilder();
		$qb2->select('u')
			->from(User::class, 'u')
			->join('u.roles', 'r')
			->where($qb2->expr()->in('r.id', $arr));

		return array_map(static fn(User $u) => $u->getEmail(), $qb2->getQuery()->getResult());
	}

	public function checkUserToken(string $token, string $email): bool
	{
		$user      = $this->getByEmail($email);
		$tokenHash = md5($token);

		if (!$user) {
			return false;
		}

		$action = $user->getUserActionByToken($tokenHash);

		if (!$action || $action->isDone()) {
			return false;
		}

		$dateCreated = $action->getCreated();
		$dateLimit   = new DateTime;
		$dateLimit->sub(new DateInterval('P0DT24H'));
		if ($dateCreated < $dateLimit) {
			return false;
		}

		if ($action->getType() !== UserAction::RESET_PASSWORD) {
			return false;
		}

		return true;
	}

	public function getAllNonAdminUsersBasicData(): array
	{
		if ($this->cAllNonAdminUsers === null) {
			$this->cAllNonAdminUsers = [];

			$qb = $this->em->getRepository(User::class)->createQueryBuilder('u')
				->select('u.id, u.name, u.lastname, u.email')
				->leftJoin('u.roles', 'r')
				->where('r.ident NOT IN (:notRoles) OR r.ident IS NULL')
				->andWhere('u.isActive = 1')
				->setParameters([
					'notRoles' => [Role::ADMIN, Role::SUPERADMIN],
				]);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$this->cAllNonAdminUsers[$row['id']] = [
					'name'      => trim($row['name'] . ' ' . $row['lastname'] . ' (' . $row['email'] . ')'),
					'firstName' => (string) $row['name'],
					'lastName'  => (string) $row['lastname'],
					'email'     => (string) $row['email'],
				];
			}
		}

		return $this->cAllNonAdminUsers;
	}

	/**
	 * @return User[]
	 */
	public function getAllAdmins(): array
	{
		if ($this->cAllAdmins === null) {
			$this->cAllAdmins = [];

			$expr = $this->em->getExpressionBuilder();

			foreach ($this->em->getRepository(User::class)->createQueryBuilder('u')
				         ->innerJoin('u.roles', 'r')
				         ->where(
					         $expr->in(
						         'r.id',
						         $this->em->createQueryBuilder()->select('IDENTITY(acl.role)')
							         ->from(Acl::class, 'acl')
							         ->innerJoin('acl.privilege', 'aclP', Join::WITH, 'aclP.name = :aclPName')
							         ->innerJoin('acl.resource', 'aclR', Join::WITH, 'aclR.name = :aclRName')->getDQL(),
					         ),
				         )
				         ->setParameters([
					         'aclPName' => 'access',
					         'aclRName' => 'Core:Admin',
				         ])
				         ->getQuery()->getResult() as $row) {
				/** @var User $row */
				$this->cAllAdmins[$row->getId()] = $row;
			}
		}

		return $this->cAllAdmins;
	}

	/**
	 * @return array<int, string>
	 */
	public function getAllAdminsForSelect(): array
	{
		$result = [];

		foreach ($this->getAllAdmins() as $row) {
			$result[$row->getId()] = $row->getFullName();
		}

		return $result;
	}
}
