<?php declare(strict_types = 1);

namespace Users\Model;

use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Sites;
use Nette\Localization\ITranslator;
use Users\Model\Entities\User;

class UsersMailer
{
	/** @var Sites */
	protected $sites;

	/** @var MailBuilderFactory */
	protected $mailer;

	/** @var ITranslator */
	protected $translator;

	public function __construct(Sites $sites, MailBuilderFactory $mailer, ITranslator $translator)
	{
		$this->sites      = $sites;
		$this->mailer     = $mailer;
		$this->translator = $translator;
	}

	public function sendNewUser(User $user, string $token): void
	{
		$mail = $this->mailer->create();

		$site    = $this->sites->getCurrentSite();
		$subject = $this->translator->translate('users.emails.createUser');
		$mail->setSubject($subject);
		$mail->setParameters([
			'subject' => $subject,
			'email'   => $user->getEmail(),
			'token'   => $token,
		]);

		$file = TEMPLATES_DIR . '/Front/default/Users/_emails/newUser.latte';

		if (file_exists($file)) {
			$mail->setTemplateFile($file);
			$mail->setFrom($site->getEmail(), $site->getSiteName());
			$mail->addTo($user->getEmail());
			$mail->send();
		}
	}

	public function sendForgotPasswordLink(string $email, string $token): void
	{
		$mail = $this->mailer->create();
		$site = $this->sites->getCurrentSite();

		$subject = $this->translator->translate('usersFront.forgotPasswordForm.subject', ['siteName' => $site->getSiteName()]);
		$mail->setSubject($subject);
		$mail->setParameters([
			'subject' => $subject,
			'email'   => $email,
			'token'   => $token,
		]);

		$file = TEMPLATES_DIR . '/Front/default/Users/_emails/forgotPassword.latte';

		$mail->setTemplateFile(file_exists($file) ? $file : __DIR__ . '/../FrontModule/Components/forgotPasswordEmail.latte');
		$mail->setFrom($site->getEmail(), $site->getSiteName());
		$mail->addTo($email);
		$mail->send();
	}
}
