<?php declare(strict_types=1);

namespace Weather\Model\WeatherHandlers;

use Cmfcmf\OpenWeatherMap;
use Nette\Http\Request;
use Http\Factory\Guzzle\RequestFactory;
use Http\Adapter\Guzzle6\Client as GuzzleAdapter;
use Weather\Model\Entities\CurrentWeather;
use Weather\Model\Entities\Forecast;
use Weather\Model\Entities\WeatherForecast;
use Weather\Model\Utils\Temperature;
use Weather\Model\Utils\Unit;
use Weather\Model\Utils\Weather;

class OpenWeatherMapHandler extends BaseWeatherHandler implements IWeatherHandler {

	/** @var Request */
	protected $httpRequest;

	/**
	 * OpenWeatherMapHandler constructor.
	 * @param Request $httpRequest
	 */
	public function __construct(Request $httpRequest) {
		$this->httpRequest = $httpRequest;
	}

	/**
	 * @inheritDoc
	 */
	public function getIdent(): string {
		return 'openWeatherMap';
	}

	/**
	 * @inheritDoc
	 */
	public function getWeather(string $destination): CurrentWeather {
		$units = $this->getConfig()['units'];
		$lang = $this->lang;

		$weatherInstance = $this->getInstance()->getWeather($destination, $units, $lang);

		$weather = new Weather;
		$weather->icon = $weatherInstance->weather->icon;
		$weather->id = $weatherInstance->weather->id;
		$weather->description = $weatherInstance->weather->description;

		$temperature = new Temperature;

		if ($weatherInstance->temperature->night !== null) {
			$temperature->night = new Unit(
				$weatherInstance->temperature->night->getUnit(),
				$weatherInstance->temperature->night->getValue(),
				$weatherInstance->temperature->night->getDescription(),
			);
		}

		if ($weatherInstance->temperature->day !== null) {
			$temperature->day = new Unit(
				$weatherInstance->temperature->day->getUnit(),
				$weatherInstance->temperature->day->getValue(),
				$weatherInstance->temperature->day->getDescription(),
			);
		}

		$currentWeather = new CurrentWeather;
		$currentWeather->weather = $weather;
		$currentWeather->temperature = $temperature;

		return $currentWeather;
	}

	/**
	 * @inheritDoc
	 */
	public function getWeatherForecast(string $destination, int $days): WeatherForecast {
		$units = $this->getConfig()['units'];
		$lang = $this->lang;

		$weatherForecast = $this->getInstance()->getWeatherForecast($destination, $units, $lang, $this->getConfig()['apiKey'], $days);

		$wf = new WeatherForecast;
		foreach ($weatherForecast as $forecast) {
			$temperature = new Temperature;
			$temperature->day = new Unit(
				$forecast->temperature->day->getUnit(),
				$forecast->temperature->day->getValue(),
				$forecast->temperature->day->getDescription(),
			);
			$temperature->night = new Unit(
				$forecast->temperature->night->getUnit(),
				$forecast->temperature->night->getValue(),
				$forecast->temperature->night->getDescription(),
			);

			$weather = new Weather;
			$weather->description = $forecast->weather->description;
			$weather->id = $forecast->weather->id;
			$weather->icon = $forecast->weather->icon;
			$weather->date = $forecast->time->day;

			$f = new Forecast;
			$f->temperature = $temperature;
			$f->weather = $weather;

			$wf->addForecast($f);
		}

		return $wf;
	}

	/**
	 * @return OpenWeatherMap
	 */
	private function getInstance(): OpenWeatherMap {
		$apiKey = $this->getConfig()['apiKey'];
		$httpRequestFactory = new RequestFactory();
		$httpClient = GuzzleAdapter::createWithConfig([]);

		return new OpenWeatherMap($apiKey, $httpClient, $httpRequestFactory);
	}

}