<?php declare(strict_types=1);

namespace Weather\Model\WeatherHandlers;

use Nette\Utils\DateTime;
use Weather\Model\Entities\CurrentWeather;
use Weather\Model\Entities\Forecast;
use Weather\Model\Entities\WeatherForecast;
use Weather\Model\Utils\Temperature;
use Weather\Model\Utils\Unit;
use Weather\Model\Utils\Weather;
use Yr\Location;
use Yr\Yr;

class YrNoHandler extends BaseWeatherHandler implements IWeatherHandler
{
	/**
	 * @return string
	 */
	public function getIdent(): string {
		return 'yrNo';
	}

	/**
	 * @inheritDoc
	 */
	public function getWeather(string $destination): CurrentWeather {
		$location = $this->getInstance($destination);
		$weatherInstance = $location->getCurrentForecast();

		$weather = new Weather;
		$weather->icon = $weatherInstance->getSymbol('var');
		$weather->id = '';
		$weather->description = $weatherInstance->getSymbol('name');
		$weather->date = $weatherInstance->getFrom();

		$temperature = new Temperature;

		$temperature->day = new Unit(
			$weatherInstance->getTemperature('unit'),
			(float) $weatherInstance->getTemperature('value'),
			$weatherInstance->getSymbol('name'),
		);

		// TODO $temperature->night = ...

		$currentWeather = new CurrentWeather;
		$currentWeather->weather = $weather;
		$currentWeather->temperature = $temperature;

		return $currentWeather;
	}

	/**
	 * @inheritDoc
	 */
	public function getWeatherForecast(string $destination, int $days): WeatherForecast {
		$now = new DateTime;
		$from = strtotime($now->format('Y-m-d'));
		$to = strtotime($now->modifyClone(sprintf('+%d day%s', $days + 1, $days > 1 ? 's' : ''))->format('Y-m-d')) - 1/*sec*/;

		$location = $this->getInstance($destination);
		$forecasts = $location->getPeriodicForecasts($from, $to);

		// separate to day and night
		$parts = [];
		/** @var \Yr\Forecast $forecast */
		foreach ($forecasts as $forecast) {

			// find night info
			if ($forecast->getPeriod() == '0') {
				$parts['night'][] = $forecast;
			}

			// find day info
			if ($forecast->getPeriod() == '2') {
				$parts['day'][] = $forecast;
			}
		}

		$weatherForecast = new WeatherForecast;
		for ($i = 0; $i < $days; $i++) {

			/** @var \Yr\Forecast $dayForecast */
			$dayForecast = $parts['day'][$i];

			/** @var \Yr\Forecast $nightForecast */
			$nightForecast = $parts['night'][$i];

			if (!$dayForecast) {
				continue;
			}

			$temperature = new Temperature;
			$temperature->day = new Unit(
				$dayForecast->getTemperature('unit'),
				(float) $dayForecast->getTemperature('value'),
				$dayForecast->getSymbol('name'),
			);
			$temperature->night = new Unit(
				$nightForecast->getTemperature('unit'),
				(float) $nightForecast->getTemperature('value'),
				$nightForecast->getSymbol('name'),
			);

			$weather = new Weather;
			$weather->description = $dayForecast->getSymbol('name');
			$weather->id = '';
			$weather->icon = $dayForecast->getSymbol('var');
			$weather->date = $dayForecast->getFrom();

			$forecast = new Forecast;
			$forecast->temperature = $temperature;
			$forecast->weather = $weather;

			$weatherForecast->addForecast($forecast);
		}

		return $weatherForecast;
	}

	/**
	 * @param string $destination
	 * @return Location
	 */
	public function getInstance(string $destination): Location {
		$tempDir = $this->container->getParameters()['tempDir'];
		return Yr::create($destination, $tempDir, 10, $this->lang);
	}
}