<?php declare(strict_types = 1);

namespace Zasilkovna\AdminModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use Nette\Localization\ITranslator;
use Zasilkovna\Model\OrderApiService;
use Zasilkovna\Model\OrdersExported;

class ZasilkovnaOrdersSubscriber implements EventSubscriber
{
	/** @var OrdersExported */
	protected $ordersExported;

	/** @var OrderApiService */
	protected $apiService;

	/** @var ITranslator */
	protected $translator;

	public function __construct(OrdersExported $ordersExported, OrderApiService $apiService, ITranslator $translator)
	{
		$this->ordersExported = $ordersExported;
		$this->apiService     = $apiService;
		$this->translator     = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => 'ordersGridAttached',
		];
	}

	public function ordersGridAttached(ControlEvent $event)
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('zasilkovna', 'zasilkovna.ordersGrid.exportApi', function(array $ids) use ($control) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);
			if (!$orders) {
				$control->getPresenter()->flashMessageSuccess($this->translator->translate('zasilkovna.export.noOrders'));
				$control->getPresenter()->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->generateOrderOverApi($orders);

			if ($result['error'] === 0)
				$control->getPresenter()->flashMessageSuccess($this->translator->translate('zasilkovna.export.success'));
			else if ($result['error'] > 0 && $result['ok'] > 0)
				$control->getPresenter()->flashMessageWarning($this->translator->translate('zasilkovna.export.partial', null, $result));
			else
				$control->getPresenter()->flashMessageDanger($this->translator->translate('zasilkovna.export.error'));

			$control->getPresenter()->redrawControl('flashes');
		});
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		$event->addNavigation(
			'zasilkovna.ordersNavigation.exportZasilkovna',
			['navigationCallback!', ['key' => 'zasilkovnaExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'zasilkovnaExport')
			return;

		$orders = $this->ordersExported->getOrdersNotExported();
		if (!$orders) {
			$event->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->generateOrderOverApi($orders);

		if ($result['error'] === 0)
			$event->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.export.success'));
		else if ($result['error'] > 0 && $result['ok'] > 0)
			$event->presenter->flashMessageWarning($this->translator->translate('zasilkovna.export.partial', null, $result));
		else
			$event->presenter->flashMessageDanger($this->translator->translate('zasilkovna.export.error'));

		$event->presenter->redrawControl('flashes');
	}
}
