<?php declare(strict_types = 1);

namespace Zasilkovna\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\Entities\Spedition;
use Kdyby\Doctrine\EntityManager;
use Nette\Localization\ITranslator;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;

class OrderFormSubscriber implements EventSubscriber
{
	/** @var array */
	protected $conf;

	/** @var EntityManager */
	protected $em;

	/** @var ITranslator */
	protected $translator;

	public function __construct(array $conf, EntityManager $em, ITranslator $translator)
	{
		$this->conf       = $conf;
		$this->em         = $em;
		$this->translator = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'   => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'  => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate' => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'   => ['orderBeforeSave', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$event->template->zasilkovnaApiKey = $this->conf['apiKey'];

		$form->addText('zasilkovnaName', 'zasilkovna.orderForm.zasilkovnaName')
			->setPlaceholder('zasilkovna.orderForm.selectDispensingPoint')
			->getControlPrototype()->class[] = 'orderFormZasilkovnaName';

		$form->addHidden('zasilkovnaId')->getControlPrototype()->class[] = 'orderFormZasilkovnaId';
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data = $event->data;
		$form = $event->form;

		if ($form->getComponent('zasilkovnaName', false))
			$form['zasilkovnaName']->setDefaultValue($data['zasilkovnaName']);
		if ($form->getComponent('zasilkovnaId', false))
			$form['zasilkovnaId']->setDefaultValue($data['zasilkovnaId']);
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form         = $event->form;
		$values       = $form->getValues();
		$speditionIds = $this->getZasikovnaSpeditionIds();

		if (in_array($values['spedition'], $speditionIds) && !$values['zasilkovnaId'])
			$form['zasilkovnaName']->addError($this->translator->translate('zasilkovna.orderForm.zasilkovnaIdMissing'));
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		if (in_array($values['spedition'], $this->getZasikovnaSpeditionIds())) {
			$entity             = new ZasilkovnaOrder($event->order, $values['zasilkovnaId']);
			$entity->parcelName = $values['zasilkovnaName'];
			$this->em->persist($entity);
		}
	}

	protected function getZasikovnaSpeditionIds(): array
	{
		$result = [];
		foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')->select('s.id, s.ident')
			         ->where('s.ident = :ident')->setParameter('ident', 'zasilkovna')
			         ->getQuery()->useResultCache(true, 30)->getArrayResult() as $row)
			$result[$row['ident']] = $row['id'];

		return $result;
	}
}
