<?php declare(strict_types = 1);

namespace Zasilkovna\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\Entities\Spedition;
use Kdyby\Doctrine\EntityManager;
use Nette\Http\Request;
use Nette\Localization\ITranslator;
use Zasilkovna\Model\Branches;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;

class OrderFormSubscriber implements EventSubscriber
{
	/** @var array */
	protected $conf;

	/** @var EntityManager */
	protected $em;

	/** @var ITranslator */
	protected $translator;

	/** @var Branches */
	protected $branches;

	/** @var Request */
	protected $httpRequest;

	public function __construct(array $conf, EntityManager $em, ITranslator $translator, Branches $branches, Request $request)
	{
		$this->conf        = $conf;
		$this->em          = $em;
		$this->translator  = $translator;
		$this->httpRequest = $request;
		$this->branches    = $branches;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$event->template->zasilkovnaApiKey = $this->conf['apiKey'];

		$form->addText('zasilkovnaName', 'zasilkovna.orderForm.zasilkovnaName')
			->setPlaceholder('zasilkovna.orderForm.selectDispensingPoint')
			->getControlPrototype()->class[] = 'orderFormZasilkovnaName';

		$form->addHidden('zasilkovnaId')->getControlPrototype()->class[] = 'orderFormZasilkovnaId';
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data    = $event->data;
		$form    = $event->form;
		$value   = $form->getValues();
		$request = $this->httpRequest;

		if ($form->getComponent('zasilkovnaName', false))
			$form['zasilkovnaName']->setDefaultValue($data['zasilkovnaName']);
		if ($form->getComponent('zasilkovnaId', false))
			$form['zasilkovnaId']->setDefaultValue($data['zasilkovnaId']);

		$spedition = $request->getPost('spedition');
		if (!in_array($spedition, $this->getZasikovnaSpeditionIds()))
			return;

		$branch = $this->branches->getBranch((int) $data['zasilkovnaId']);
		if (!$branch)
			return;

		foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
			$form[$k . '2']->setValue($form[$k . '2']->isRequired() && !$value[$k] ? 'placeholder' : ($value[$k] ?: null));
		}

		$form['useAddrDeli']->setValue(true);
		$form['street2']->setValue($branch['street']);
		$form['city2']->setValue($branch['city']);
		$form['postal2']->setValue($branch['zip']);
		$form['country2']->setValue(strtolower($branch['country']));
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$spedition = $request->getPost('spedition');

		if (in_array($spedition, $this->getZasikovnaSpeditionIds())) {
			foreach (['zasilkovnaId', 'zasilkovnaName'] as $k) {
				$data[$k] = $request->getPost($k);
			}

			$data['disableDeliveryAddressSpedition'] = 'zasilkovna';
			$data['disableDeliveryAddress']          = true;
		} else if ($data['disableDeliveryAddressSpedition'] == 'zasilkovna') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$spedition = $request->getPost('spedition');
		if (!in_array($spedition, $this->getZasikovnaSpeditionIds()))
			return;

		$branch = $this->branches->getBranch((int) $data['zasilkovnaId']);
		if (!$branch)
			return;

		foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k)
			$data[$k . '2'] = $data[$k];

		$data['useAddrDeli'] = true;
		$data['street2']     = $branch['street'];
		$data['city2']       = $branch['city'];
		$data['postal2']     = $branch['zip'];
		$data['country2']    = strtoupper($branch['country']);
	}


	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form         = $event->form;
		$values       = $form->getValues();
		$speditionIds = $this->getZasikovnaSpeditionIds();

		if (in_array($values['spedition'], $speditionIds) && !$values['zasilkovnaId'])
			$form['zasilkovnaName']->addError($this->translator->translate('zasilkovna.orderForm.zasilkovnaIdMissing'));
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		if (in_array($values['spedition'], $this->getZasikovnaSpeditionIds())) {
			$entity             = new ZasilkovnaOrder($event->order, $values['zasilkovnaId']);
			$entity->parcelName = $values['zasilkovnaName'];
			$this->em->persist($entity);
		}
	}

	protected function getZasikovnaSpeditionIds(): array
	{
		$result = [];
		foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')->select('s.id, s.ident')
			         ->where('s.ident = :ident')->setParameter('ident', 'zasilkovna')
			         ->getQuery()->useResultCache(true, 30)->getArrayResult() as $row)
			$result[$row['ident']] = $row['id'];

		return $result;
	}
}
