<?php declare(strict_types = 1);

namespace Zasilkovna\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Listeners\CarrierListener;
use Nette\Utils\DateTime;
use Zasilkovna\Model\ZasilkovnaConfig;

#[ORM\Entity]
#[ORM\EntityListeners([CarrierListener::class])]
#[ORM\Table(name: 'zasilkovna__order')]
class ZasilkovnaOrder
{
	public const STATUS_COMPLETED = 'completed';
	public const STATUS_NOT_FOUND = 'notFound';

	#[ORM\Id]
	#[ORM\OneToOne(targetEntity: Order::class)]
	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	protected Order $order;

	#[ORM\Column(type: 'string', nullable: true)]
	protected ?string $parcelId = null;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $carrierId = null;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $carrierPickupPointId = null;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $parcelName = null;

	/**
	 * @var DateTimeInterface|null
	 */
	#[ORM\Column(type: 'datetime', nullable: true)]
	private $exported;

	#[ORM\Column(type: 'string', nullable: true)]
	private ?string $exportType = null;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $numberPackage = null;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $lastStatus = null;

	/**
	 * @var int|string|null
	 */
	#[ORM\Column(type: 'bigint', nullable: true)]
	public $idPackage;

	#[ORM\Column(type: 'integer', nullable: true, options: ['unsigned' => true])]
	public ?int $serviceId = null;

	/**
	 * @var Collection<ZasilkovnaParcelNumber>
	 */
	#[ORM\OneToMany(targetEntity: ZasilkovnaParcelNumber::class, mappedBy: 'parcelOrder', cascade: ['all'], orphanRemoval: true, indexBy: 'numberPackage')]
	public Collection $associatedNumberPackages;

	public function __construct(Order $order, ?string $parcelId = null)
	{
		$this->order                    = $order;
		$this->parcelId                 = $parcelId;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function getParcelId(): ?string { return $this->parcelId; }

	/**
	 * @param int|string $parcel
	 */
	public function setParcelId($parcel): void { $this->parcelId = (string) $parcel; }

	public function export(string $type, ?DateTime $datetime = null): self
	{
		$this->exportType = $type;
		$this->exported   = $datetime ?? new DateTime;

		return $this;
	}

	public function getExportType(): ?string { return $this->exportType; }

	public function getExported(): ?DateTimeInterface { return $this->exported; }

	public function getTrackingUrl(): string
	{
		return sprintf(ZasilkovnaConfig::loadString('trackingUrl') ?: '', $this->numberPackage);
	}

	/**
	 * @return string[]
	 */
	public function getTrackingUrls(): array
	{
		$result = [];
		if ($this->getTrackingUrl() !== '' && $this->getTrackingUrl() !== '0') {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(ZasilkovnaConfig::loadString('trackingUrl') ?: '', $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->idPackage     = null;
		$this->lastStatus    = null;
		$this->exported      = null;
		$this->associatedNumberPackages->clear();
	}

	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(ZasilkovnaParcelNumber $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}
}
