<?php declare(strict_types = 1);

namespace Zasilkovna\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use EshopOrders\Model\ExpeditionLogger;
use EshopOrders\Model\Helpers\OrderHelper;
use Exception;
use Nette\Utils\FileSystem;
use SoapClient;
use SoapFault;
use Tracy\Debugger;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;

class OrderApiService
{
	public function __construct(
		protected array                  $zasilkovna,
		protected EntityManagerDecorator $em,
		protected ExpeditionLogger       $expeditionLogger,
	)
	{
	}

	/**
	 * @param ZasilkovnaOrder[] $orders
	 *
	 * @throws SoapFault
	 */
	public function generateOrderOverApi(array $orders, int $quantity = 1): array
	{
		$gw          = new SoapClient("http://www.zasilkovna.cz/api/soap.wsdl");
		$apiPassword = $this->zasilkovna['apiPassword'];
		$result      = [
			'ok'    => 0,
			'error' => 0,
		];

		foreach ($orders as $zo) {
			/** @var ZasilkovnaOrder $zo */
			$order = $zo->getOrder();

			if (!$order->getId()) {
				continue;
			}

			$data = [];

			try {
				$address = $order->getAddressDelivery();
				if (!$address) {
					continue;
				}

				$cod = 0;

				if ($order->getPaymentIdent() === 'cod') {
					$curr = $order->getCurrencyCode();
					$cod  = round($order->getPrice(true), $curr === 'CZK' ? 0 : ($order->currency?->decimals ?: 2));

					if (
						($curr === 'EUR')
						|| (Strings::lower((string) $address->getCountry()?->getId() ?: 'cz') === 'sk')
					) {
						$cod = OrderHelper::roundSkCod($cod);
					}
				}

				$data = [
					'number'   => (string) $order->getId(),
					'name'     => $address->getFirstName(),
					'surname'  => $address->getLastName(),
					'email'    => $address->getEmail(),
					'phone'    => $address->getPhone(),
					'cod'      => $cod,
					'value'    => round($order->getPriceItems(true), $order->getCurrencyCode() === 'CZK' ? 0 : 2),
					'currency' => $order->getCurrencyCode(),
					'eshop'    => $this->zasilkovna['eshopName'],
					'weight'   => $order->getItemsWeight() ?: 1,
				];

				if ($zo->carrierId && $zo->carrierPickupPointId) {
					$data['addressId']          = $zo->carrierId;
					$data['carrierPickupPoint'] = $zo->carrierPickupPointId;
					$packet                     = $gw->createPacket($apiPassword, $data);
				} else if ($zo->getParcelId()) {
					$data['addressId'] = $zo->getParcelId();
					$packet            = $gw->createPacket($apiPassword, $data);
				} else {
					$address = $order->getAddressDelivery();

					if ($address) {
						$zip          = str_replace([' ', '-'], '', $address->getPostal() ?: '');
						$lowerCountry = strtolower($address->getCountry()?->getId() ?: 'CZ');

						if ($lowerCountry === 'pl') {
							$zip = substr($zip, 0, 2) . '-' . substr($zip, 2);
						} else if (!Arrays::contains(['de'], $lowerCountry)) {
							$zip = substr($zip, 0, 3) . ' ' . substr($zip, 3);
						}

						$data['addressId'] = $zo->serviceId;
						$data['street']    = $address->getStreet();
						$data['city']      = $address->getCity();
						$data['zip']       = $zip;
					}

					$packet = $gw->createPacket($apiPassword, $data);
				}

				$zo->export('api');
				$zo->idPackage     = $packet->id;
				$zo->numberPackage = $packet->barcode;
				$this->em->persist($zo);
				$this->em->flush();
				$result['ok']++;
			} catch (SoapFault $e) {
				Debugger::log(json_encode($data), 'zasilkovnaApi');
				Debugger::log(json_encode($e->detail), 'zasilkovnaApi');

				if (isset($e->detail->PacketAttributesFault)) {
					$err = $e->detail->PacketAttributesFault->attributes->fault->name . ' - ' . $e->detail->PacketAttributesFault->attributes->fault->fault;
					$this->expeditionLogger->logError('zasilkovna', $err, $order->getId());
				}

				$result['error']++;
			} catch (Exception $e) {
				Debugger::log(json_encode($data), 'zasilkovnaApi');
				Debugger::log($e->getMessage(), 'zasilkovnaApi');
				$this->expeditionLogger->logError('zasilkovna', $e->getMessage(), $order->getId());

				$result['error']++;
			}
		}

		return $result;
	}

	public function generateLabelPdf(array $orders, bool $sendToPrinter = false): array
	{
		$gw = new SoapClient("http://www.zasilkovna.cz/api/soap.wsdl");

		$apiPassword = $this->zasilkovna['apiPassword'];
		$format      = $this->zasilkovna['format'];

		$result = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		$packageIds = [];
		$ordersIds  = [];
		foreach ($orders as $order) {
			/** @var ZasilkovnaOrder $order */

			$ordersIds[] = $order->getOrder()->getId();

			if ($order->idPackage) {
				$packageIds[] = $order->idPackage;
			}
		}

		if (count($packageIds) > 0) {
			try {
				$pdf = $gw->packetsLabelsPdf($apiPassword, $packageIds, $format, 0);
				FileSystem::createDir(TMP_DIR . '/zasilkovnaLabels');
				if (file_put_contents(TMP_DIR . '/zasilkovnaLabels/stitky.pdf', $pdf)) {
					$result['files'][] = TMP_DIR . '/zasilkovnaLabels/stitky.pdf';
					$result['ok']++;
				} else {
					$result['error']++;
				}
			} catch (SoapFault $e) {
				Debugger::log($e, 'zasilkovnaApi');
				$this->expeditionLogger->logError('zasilkovna', $e->getMessage(), (int) $ordersIds[0]);

				$result['error']++;
			}
		}

		return $result;
	}

	public function checkCompleted(ZasilkovnaOrder $zo): bool
	{
		$gw = new SoapClient("http://www.zasilkovna.cz/api/soap.wsdl");

		$apiPassword = $this->zasilkovna['apiPassword'];

		try {
			$status = $gw->packetStatus($apiPassword, $zo->idPackage);

			return $status->codeText === 'delivered';
		} catch (Exception $e) {
			if (\str_starts_with($e->getMessage(), 'Incorrect packet')) {
				$this->em->getConnection()->executeStatement("UPDATE zasilkovna__order SET last_status = '" . ZasilkovnaOrder::STATUS_NOT_FOUND . "' WHERE order_id = " . $zo->getOrder()->getId());
			} else {
				Debugger::log('Complete check error - ' . $e->getMessage(), 'zasilkovna');
			}
		}

		return false;
	}
}
