<?php declare(strict_types = 1);

namespace Zasilkovna\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;

/**
 * @method ZasilkovnaOrder|null getReference($id)
 * @method ZasilkovnaOrder[] getAll()
 * @method ZasilkovnaOrder|null get($id)
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = ZasilkovnaOrder::class;

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return ZasilkovnaOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @param int[]|string[] $ids
	 * @param bool           $loadFull
	 *
	 * @return ZasilkovnaOrder[]
	 */
	public function getOrdersExported(array $ids = [], bool $loadFull = true): array
	{
		return $this->getOrders(true, $ids, $loadFull);
	}

	/**
	 * @return ZasilkovnaOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->getEr()->createQueryBuilder('zo')
			         ->where('zo.lastStatus NOT IN (:status) OR zo.lastStatus IS NULL')
			         ->andWhere('zo.numberPackage IS NOT NULL')
			         ->setParameters(new ArrayCollection([
				         new Parameter('status', [
					         ZasilkovnaOrder::STATUS_COMPLETED,
					         ZasilkovnaOrder::STATUS_NOT_FOUND,
				         ]),
			         ]))->getQuery()
			         ->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return ZasilkovnaOrder[]
	 */
	public function getOrders(?bool $isExported = null, array $ids = [], bool $loadFull = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('zo')
			->addSelect('anp')
			->leftJoin('zo.associatedNumberPackages', 'anp')
			->orderBy('zo.order')->groupBy('zo.order');

		if ($loadFull) {
			$qb->addSelect('o, zo, oad, oai, os, s, oCurrency')
				->join('zo.order', 'o')
				->leftJoin('o.currency', 'oCurrency')
				->leftJoin('o.addressDelivery', 'oad')
				->leftJoin('o.addressInvoice', 'oai')
				->leftJoin('o.spedition', 'os')
				->leftJoin('os.spedition', 's');
		}

		if ($isExported === true) {
			$qb->where('zo.exported IS NOT NULL');
		} else if ($isExported === false) {
			$qb->where('zo.exported IS NULL');
		}

		if (!empty($ids)) {
			$qb->andWhere('zo.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	/**
	 * @param int[]|string[] $orderIds
	 */
	public function markAsCompleted(array $orderIds): void
	{
		$this->em->getConnection()->executeStatement("UPDATE zasilkovna__order SET last_status = '" . ZasilkovnaOrder::STATUS_COMPLETED . "' WHERE order_id IN (" . implode(',', $orderIds) . ")");
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->getEr()->createQueryBuilder('z')
			         ->select('IDENTITY(z.order) as id')
			         ->leftJoin('z.associatedNumberPackages', 'acp')
			         ->orWhere('z.numberPackage LIKE :number')
			         ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
