<?php declare(strict_types = 1);

namespace Zasilkovna\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Exception;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;
use Zasilkovna\Model\Branches;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;
use Zasilkovna\Model\Entities\ZasilkovnaParcelNumber;
use Zasilkovna\Model\OrdersExported;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var ZasilkovnaOrder[]|null */
	protected static ?array          $allOrders = null;
	protected OrdersExported         $ordersExported;
	protected EntityManagerDecorator $em;
	protected Branches               $branches;
	protected EventDispatcher        $eventDispatcher;

	public function __construct(
		OrdersExported  $ordersExported, EntityManagerDecorator $em, Branches $branches,
		EventDispatcher $eventDispatcher
	)
	{
		$this->ordersExported  = $ordersExported;
		$this->em              = $em;
		$this->branches        = $branches;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::createForm'    => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'      => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'zasilkovna') {
			$control->template->zasilkovnaExport            = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$export         = null;
		$data           = $event->data;
		$speditionIdent = 'zasilkovna';

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []), false);
			}

			$export = self::$allOrders[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrdersExported([$order->getId()]);
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$zasilkovnaIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'zasilkovna') {
				$zasilkovnaIds[] = $sped->getId();
			}
		}

		if (empty($zasilkovnaIds)) {
			return;
		}

		$container = $form->addContainer('zasilkovna', 'zas');
		$container->addText('parcelId', 'zasilkovna.entity.parcelId')
			->setDescription('zasilkovna.entity.parcelsList');
		$container->addText('carrierId', 'zasilkovna.entity.carrierId')
			->setDescription('zasilkovna.entity.carrierIdDescription');
		$container->addText('numberPackage', 'zasilkovna.entity.numberPackage')
			->setDescription('zasilkovna.entity.numberPackageDesc');
		$container->addText('idPackage', 'zasilkovna.entity.idPackage')
			->setDescription('zasilkovna.entity.idPackageDescription');
		$container->addBool('removeExport', 'zasilkovna.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'zasilkovna.entity.updateDeliveryAddress')
			->setDefaultValue(1);

		$form->getComponent('spedition')->addCondition($form::IS_IN, $zasilkovnaIds)
			->toggle($form['zasilkovna']['parcelId']->getHtmlId())
			->toggle($form['zasilkovna']['numberPackage']->getHtmlId())
			->toggle($form['zasilkovna']['idPackage']->getHtmlId())
			->toggle($form['zasilkovna']['removeExport']->getHtmlId())
			->toggle($form['zasilkovna']['updateDeliveryAddress']->getHtmlId());
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control         = $event->control;
		$orderId         = $orderSpedition->getOrder()->getId();
		$values          = $event->values;
		$zasilkovnaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isZasilkovna = false;
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'zasilkovna' && $sped->getId() == $values->spedition) {
				$isZasilkovna = true;
				break;
			}
		}

		try {
			if ($isZasilkovna) {
				$branch = null;
				if ($values->zasilkovna->parcelId !== null && $values->zasilkovna->parcelId !== '') {
					$branch = $this->branches->getBranch(
						(int) $values->zasilkovna->parcelId,
						$values->zasilkovna->carrierId ? (int) $values->zasilkovna->carrierId : null
					);

					if (!$branch) {
						$event->form->addError('zasilkovna.orderForm.branchNotFound');
						$control->redrawControl('form');

						return;
					}
				}

				if (!$zasilkovnaOrder) {
					$zasilkovnaOrder = new ZasilkovnaOrder($orderSpedition->getOrder(), $values->zasilkovna->parcelId);
				}

				$deliveryAddress = $zasilkovnaOrder->getOrder()->getAddressDelivery();
				if ($deliveryAddress && $branch && $values['zasilkovna']->updateDeliveryAddress) {
					$deliveryAddress->setStreet($branch['street']);
					$deliveryAddress->setPostal($branch['zip']);
					$deliveryAddress->setCity($branch['city']);
					$this->em->persist($deliveryAddress);
					if ($event->presenter && isset($event->presenter['orderForm'])) {
						$event->presenter['orderForm']->redrawControl('addressDelivery');
					}
				}

				$orderSpedition->deliveryPointId = $values->zasilkovna->parcelId ? (string) $values->zasilkovna->parcelId : null;
				$this->em->persist($orderSpedition);

				$zasilkovnaOrder->setParcelId($values->zasilkovna->parcelId);
				$zasilkovnaOrder->parcelName = $branch['name'];

				if ($values->zasilkovna->carrierId) {
					$zasilkovnaOrder->carrierId            = (string) $values->zasilkovna->carrierId;
					$zasilkovnaOrder->carrierPickupPointId = (string) $values->zasilkovna->parcelId;
				} else {
					$zasilkovnaOrder->carrierId            = null;
					$zasilkovnaOrder->carrierPickupPointId = null;
				}

				$numberPackages      = trim(str_replace(' ', '', (string) $values->zasilkovna->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && Strings::contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);

				$idPackage                  = trim(str_replace(' ', '', (string) $values->zasilkovna->idPackage));
				$zasilkovnaOrder->idPackage = Validators::isNone($idPackage) ? null : $idPackage;
				if ($numberPackagesCount <= 1) {

					$zasilkovnaOrder->numberPackage = $numberPackages[0] ?? null;
					$zasilkovnaOrder->associatedNumberPackages->clear();
				} else {
					$existsPackages            = $zasilkovnaOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$zasilkovnaOrder->numberPackage = $numberPackage;
						} else {
							if (in_array($numberPackage, $diffToAdd)) {
								$zasilkovnaOrder->associatedNumberPackages->add(new ZasilkovnaParcelNumber($numberPackage, $zasilkovnaOrder));
							}
						}
					}

					foreach ($diffToRemove as $p) {
						$zasilkovnaOrder->associatedNumberPackages->remove($p);
					}
				}

				if ($zasilkovnaOrder->idPackage && $zasilkovnaOrder->getExported() === null) {
					$zasilkovnaOrder->export('api');
				}

				if ($values->zasilkovna->removeExport === 1) {
					$zasilkovnaOrder->resetExport();
				}

				$this->em->persist($zasilkovnaOrder);

				$this->eventDispatcher->dispatch(new Event(['order' => $zasilkovnaOrder->getOrder(), 'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($zasilkovnaOrder) {
				$this->em->remove($zasilkovnaOrder);
			}
		} catch (Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('zasilkovna.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$zasilkovnaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$zasilkovnaOrder) {
			return;
		}

		$event->form->getComponent('zasilkovna')->setDefaults([
			'parcelId'      => $zasilkovnaOrder->getParcelId(),
			'numberPackage' => implode(', ', $zasilkovnaOrder->getAllNumberPackages()),
			'idPackage'     => $zasilkovnaOrder->idPackage,
			'carrierId'     => $zasilkovnaOrder->carrierId,
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'zasilkovna')) {
			return;
		}

		$orderId         = $order->getId();
		$zasilkovnaOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$zasilkovnaOrder) {
			return;
		}

		$zasilkovnaOrder->resetExport();
		$this->em->persist($zasilkovnaOrder);
		$this->em->flush($zasilkovnaOrder);
	}
}
