<?php declare(strict_types = 1);

namespace Zasilkovna\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use Zasilkovna\Model\OrderApiService;
use Zasilkovna\Model\OrdersExported;
use Zasilkovna\Model\ZasilkovnaConfig;

class ZasilkovnaOrdersSubscriber implements EventSubscriberInterface
{
	protected OrdersExported  $ordersExported;
	protected OrderApiService $apiService;
	protected Translator      $translator;
	protected Speditions      $speditionsService;

	public function __construct(
		OrdersExported  $ordersExported,
		OrderApiService $apiService,
		Translator      $translator,
		Speditions      $speditionsService
	)
	{
		$this->ordersExported    = $ordersExported;
		$this->apiService        = $apiService;
		$this->translator        = $translator;
		$this->speditionsService = $speditionsService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => ['ordersGridAttached', 255],
		];
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('zasilkovna', 'zasilkovna.ordersGrid.exportApi', function(array $ids, $quantity) use ($control) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);
			if (!$orders) {
				$control->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->generateOrderOverApi($orders, max(1, (int) $quantity));

			if ($result['error'] === 0) {
				$control->presenter
					->flashMessageSuccess($this->translator->translate('zasilkovna.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter
					->flashMessageWarning($this->translator->translate('zasilkovna.export.partial', null, $result));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('zasilkovna.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		});

		if (ZasilkovnaConfig::load('allowGenerateLabel', false)) {
			$control->addExportCallback('zasilkovnaLabels', 'zasilkovna.ordersGrid.labels', function(array $ids) use ($control) {
				$orders = $this->ordersExported->getOrdersExported($ids);
				if (!$orders) {
					$control->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.labels.noOrders'));
					$control->presenter->redrawControl('flashes');

					return;
				}

				set_time_limit(120);
				$result = $this->apiService->generateLabelPdf($orders);

				if ($result['error'] === 0) {
					$arr                                         = [
						'name' => 'Zásilkovna štítky',
						'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
							'file'     => TMP_DIR . '/zasilkovnaLabels/stitky.pdf',
							'filename' => 'Zasilkovna-stitky-' . (new DateTime())->format('Y-m-d') . '.pdf',
						]))),
					];
					$control->presenter->payload->fileRequests[] = $arr;
					$control->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.labels.success'));
				} else {
					$control->presenter
						->flashMessageDanger($this->translator->translate('zasilkovna.labels.error'));
				}

				$control->presenter->redrawControl('flashes');
			});
		}
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (EshopOrdersConfig::load('hideOrderGridNavExport')) {
			return;
		}

		$event->addNavigation(
			'zasilkovna.ordersNavigation.exportZasilkovna',
			['navigationCallback!', ['key' => 'zasilkovnaExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'zasilkovnaExport') {
			return;
		}

		$orders = $this->ordersExported->getOrdersNotExported();

		$ids = [];
		foreach ($orders as $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if (!$orders) {
			$event->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->generateOrderOverApi($orders);

		if ($result['error'] === 0) {
			$event->presenter->flashMessageSuccess($this->translator->translate('zasilkovna.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$event->presenter->flashMessageWarning($this->translator->translate('zasilkovna.export.partial', null, $result));
		} else {
			$event->presenter->flashMessageDanger($this->translator->translate('zasilkovna.export.error'));
		}

		$event->presenter->redrawControl('flashes');
	}
}
