<?php declare(strict_types = 1);

namespace Zasilkovna\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Dao as EshopOrdersDao;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Http\Request;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Zasilkovna\Model\Branches;
use Zasilkovna\Model\Entities\ZasilkovnaOrder;
use Zasilkovna\Model\ZasilkovnaConfig;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Translator             $translator;
	protected Branches               $branches;
	protected Request                $httpRequest;
	protected Countries              $countriesService;
	protected PaymentSpeditions      $paymentSpeditions;
	protected CartFacade             $cartFacade;

	public function __construct(
		EntityManagerDecorator $em,
		Translator             $translator,
		Branches               $branches,
		Request                $request,
		Countries              $countries,
		PaymentSpeditions      $paymentSpeditions,
		CartFacade             $cartFacade
	)
	{
		$this->em                = $em;
		$this->translator        = $translator;
		$this->httpRequest       = $request;
		$this->branches          = $branches;
		$this->countriesService  = $countries;
		$this->paymentSpeditions = $paymentSpeditions;
		$this->cartFacade        = $cartFacade;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopCheckout.orderBeforeSave'      => ['orderCheckoutBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$cart        = $this->cartFacade->getCart();
		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);

		$event->template->zasilkovnaApiKey = ZasilkovnaConfig::load('apiKey');
		$container                         = $form->addContainer('zasilkovna');

		$packetaWidgetSettings = [];
		foreach ($this->getZasikovnaSpeditionIds() as $row) {
			if (!$row->getSpedition()->isPickup) {
				continue;
			}

			$speditionId = $row->getSpedition()->getId();

			foreach ($row->getCountries() as $country) {
				$k = $country->getId() . '_' . $speditionId;
				if ($container->getComponent($k, false)) {
					continue;
				}

				$selectorTarget = $container->addHidden($k);
				$selectorTarget->getControlPrototype()
					->addClass('zasilkovnaTarget_' . $k);
				$container->addHidden($k . '_carrierId')
					->getControlPrototype()
					->addClass('zasilkovnaTarget_carrierId_' . $k);
				$container->addHidden($k . '_carrierPickupPointId')
					->getControlPrototype()
					->addClass('zasilkovnaTarget_carrierPickupPointId_' . $k);
				$selectorOpen = $container->addText($k . '_p')
					->setHtmlAttribute('readonly')
					->setPlaceholder('zasilkovna.orderForm.selectDispensingPoint');
				$selectorOpen->getControlPrototype()
					->addClass('zasilkovnaTrigger_' . $k);

				$widgetSettings = [
					'country'                    => strtolower($country->getId()),
					'selectorOpen'               => $selectorOpen->getHtmlId(),
					'selectorTarget'             => $selectorTarget->getHtmlId(),
					'carrierIdTarget'            => $container->getComponent($k . '_carrierId')->getHtmlId(),
					'carrierPickupPointIdTarget' => $container->getComponent($k . '_carrierPickupPointId')->getHtmlId(),
					'language'                   => $this->translator->getLocale(),
					'vendors'                    => [],
				];

				if ($cart->hasDisabledDeliveryBoxes()) {
					$widgetSettings['vendors'] = [['country' => $widgetSettings['country']]];
				}

				$packetaWidgetSettings[] = $widgetSettings;

				$event->template->zasilkovnaLastKey = $k;
			}
		}

		$event->template->packetaWidgetSettings = $packetaWidgetSettings;
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data  = $event->data;
		$form  = $event->form;
		$value = $form->getValues();

		foreach ($data['zasilkovna'] ?? [] as $k => $v) {
			if (isset($form['zasilkovna'][$k])) {
				$form['zasilkovna'][$k]->setDefaultValue($v);
			}
		}

		$branch = $this->getBranchByData($data);
		if (!$branch) {
			return;
		}

		foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
			$form->getComponent($k . '2')->setValue($form->getComponent($k . '2')->isRequired() && !$value[$k] ? 'placeholder' : ($value[$k] ?: null));
		}

		$spedition = $this->getZasikovnaSpeditionIds()[$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition->getSpedition()->isPickup) {
			$form->getComponent('useAddrDeli')->setValue(true);
			$form->getComponent('street2')->setValue($branch['street']);
			$form->getComponent('city2')->setValue($branch['city']);
			$form->getComponent('postal2')->setValue($branch['zip']);

			if (array_key_exists(strtolower($branch['country']), $form->getComponent('country2')->getItems())) {
				$form->getComponent('country2')->setValue(strtolower($branch['country']));
			} else {
				$form->getComponent('country2')->setValue(strtoupper($branch['country']));
			}
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;
		$post    = $request->getPost();

		$spedition = $this->getZasikovnaSpeditionIds()[$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];

			$data['zasilkovna'][$k]                           = $request->getPost('zasilkovna')[$k];
			$data['zasilkovna'][$k . '_p']                    = $request->getPost('zasilkovna')[$k . '_p'];
			$data['zasilkovna'][$k . '_carrierId']            = $request->getPost('zasilkovna')[$k . '_carrierId'];
			$data['zasilkovna'][$k . '_carrierPickupPointId'] = $request->getPost('zasilkovna')[$k . '_carrierPickupPointId'];

			if ($spedition->getSpedition()->isPickup) {
				$data['disableDeliveryAddressSpedition'] = 'zasilkovna';
				$data['disableDeliveryAddress']          = true;
				$data['useAddrDeli']                     = false;

				foreach (['street2', 'city2', 'postal2'] as $col) {
					if (isset($post[$col])) {
						$data[$col] = $post[$col];
					}
				}

				$event->form->getComponent('useAddrDeli')->setValue(true);
			} else if ($data['disableDeliveryAddressSpedition'] == 'zasilkovna') {
				unset($data['disableDeliveryAddress']);
			}
		} else if ($data['disableDeliveryAddressSpedition'] == 'zasilkovna') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		$branch = $this->getBranchByData($data);
		if ($branch) {
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$d[$k . '2'] = $data[$k];
			}
			$countryId   = $data['speditionCountry'];
			$countryText = $this->countriesService->getAllNameColumn()[$countryId] ?? $countryId;
			$d           += [
				'useAddrDeli'  => true,
				'street2'      => $branch['street'],
				'city2'        => $branch['city'],
				'postal2'      => $branch['zip'],
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form   = $event->form;
		$values = $form->getValues();

		$spedition = $this->getZasikovnaSpeditionIds()[$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition && $spedition->getSpedition()->isPickup) {
			$branchId = $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$branchId) {
				$form->addError($this->translator->translate('zasilkovna.orderForm.zasilkovnaIdMissing'));
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		$spedition = $this->getZasikovnaSpeditionIds()[$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition) {
			$entity = new ZasilkovnaOrder($event->order);

			if ($spedition->getSpedition()->isPickup) {
				$addrDeli = $event->addrDelivery;
				$addrInv  = $event->addrInvoice;

				if (!$addrDeli) {
					$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
					$event->order->setAddressDelivery($addrDeli);
				}
				$addrDeli->setFirstName($addrInv->getFirstName());
				$addrDeli->setLastName($addrInv->getLastName());
				$addrDeli->setEmail($addrInv->getEmail());
				$addrDeli->setPostal($addrInv->getPhone());

				$parcelId  = (int) $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};
				$carrierId = $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId() . '_carrierId'};
				$branch    = $this->branches->getBranch($parcelId, $carrierId ? (int) $carrierId : null);

				if ($branch) {
					$addrDeli->setStreet($branch['street']);
					$addrDeli->setCity($branch['city']);
					$addrDeli->setPostal($branch['zip']);

					$countryId = $values->speditionCountry;
					$addrDeli->setCountry($this->countriesService->get($countryId));
				}

				$this->em->persist($addrDeli);

				$entity->setParcelId((string) $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()});
				$entity->parcelName           = $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId() . '_p'};
				$entity->carrierId            = $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId() . '_carrierId'};
				$entity->carrierPickupPointId = $values->zasilkovna->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId() . '_carrierPickupPointId'};

				$event->order->getSpedition()->deliveryPointId = $entity->getParcelId();
				$this->em->persist($event->order->getSpedition());
			} else {
				$entity->serviceId = $spedition->getSpedition()->code1 ? (int) $spedition->getSpedition()->code1 : null;

				$addrDeli = $event->order->getAddressDelivery();
				if ($addrDeli->getCountry() && strtolower($addrDeli->getCountry()->getId()) == 'pl') {
					$postal = $addrDeli->getPostal();
					$addrDeli->setPostal(substr((string) $postal, 0, 2) . '-' . substr((string) $postal, 2));
					$this->em->persist($addrDeli);
				}
			}

			$this->em->persist($entity);
		}
	}

	public function orderCheckoutBeforeSave(OrderEvent $event): void
	{
		if (!$event->orderSpedition || !$event->orderSpedition->getSpedition() || !$event->orderPayment || !$event->orderPayment->getPayment() || $event->order->getId()) {
			return;
		}

		$spedition = $this->getZasikovnaSpeditionIds()[$event->orderPayment->getPaymentId() . '_' . $event->orderSpedition->getSpeditionId()] ?? null;
		if ($spedition) {
			$entity = new ZasilkovnaOrder($event->order);
			$this->em->persist($entity);
		}
	}

	/**
	 * @return EshopOrdersDao\PaymentSpedition[]
	 */
	protected function getZasikovnaSpeditionIds(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() !== 'zasilkovna') {
				continue;
			}

			$result[$row->getPayment()->getId() . '_' . $row->getSpedition()->getId()] = $row;
		}

		return $result;
	}

	public function getBranchByData(array $data): ?array
	{
		$spedition = $this->getZasikovnaSpeditionIds()[$data['payment'] . '_' . $data['spedition']] ?? null;
		if (!$spedition || !$spedition->getSpedition()->isPickup) {
			return null;
		}

		$parcelId  = (int) $data['zasilkovna'][$data['speditionCountry'] . '_' . $data['spedition']];
		$carrierId = $data['zasilkovna'][$data['speditionCountry'] . '_' . $data['spedition'] . '_carrierId'] ?? null;
		$branch    = $this->branches->getBranch($parcelId, $carrierId ? (int) $carrierId : null);

		return $branch ?? null;
	}

}
